#!/usr/bin/env python3
"""
Graph converter for Social Golfers problem.
Created using subagent_prompt.md version: v_02

This problem is about arranging golfers into groups over multiple rounds
such that no two golfers play together more than once.
Key challenges: Combinatorial explosion, conflict resolution, and symmetry breaking.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the social golfers problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the scheduling and conflict structure
    - Golfers are variables (type 0) that need placement
    - Constraints ensure proper grouping and no repeat pairings
    - Round/group slots are resources (type 2)
    - Individual constraints model all-different, pairing restrictions
    """
    n_groups = json_data.get('n_groups', 1)
    n_per_group = json_data.get('n_per_group', 1) 
    n_rounds = json_data.get('n_rounds', 1)
    
    n_golfers = n_groups * n_per_group
    
    G = nx.Graph()
    
    # Golfer nodes (type 0) - variables that need assignment
    # Weight based on their constraint density - central golfers more constrained
    for g in range(1, n_golfers + 1):
        # More rounds means more constraints per golfer
        constraint_density = min(n_rounds / 10.0, 1.0)  # Normalized difficulty
        G.add_node(f'golfer_{g}', type=0, weight=constraint_density)
    
    # Round-Group slot nodes (type 2) - resources that golfers compete for
    for r in range(1, n_rounds + 1):
        for gr in range(1, n_groups + 1):
            # Slots become more valuable as problem gets tighter
            scarcity = min(n_golfers / (n_groups * n_per_group * n_rounds), 1.0)
            G.add_node(f'slot_r{r}_g{gr}', type=2, weight=scarcity)
    
    # All-different constraint nodes for each round (type 1)
    for r in range(1, n_rounds + 1):
        # Weight by scope - larger scope means harder constraint
        scope_weight = min(n_golfers / 10.0, 1.0)
        G.add_node(f'alldiff_round_{r}', type=1, weight=scope_weight)
        
        # Connect all golfers to this constraint
        for g in range(1, n_golfers + 1):
            G.add_edge(f'golfer_{g}', f'alldiff_round_{r}', weight=1.0)
    
    # Group size constraints - each group must have exactly n_per_group golfers
    for r in range(1, n_rounds + 1):
        for gr in range(1, n_groups + 1):
            # Tightness based on exact cardinality requirement
            tightness = min(n_per_group / n_golfers, 1.0) if n_golfers > 0 else 0.5
            constraint_id = f'group_size_r{r}_g{gr}'
            G.add_node(constraint_id, type=1, weight=tightness)
            
            # Connect to the resource slot
            G.add_edge(constraint_id, f'slot_r{r}_g{gr}', weight=1.0)
    
    # Pairing constraint nodes - no two golfers can play together twice
    pair_count = 0
    max_pairs = (n_golfers * (n_golfers - 1)) // 2
    
    for g1 in range(1, n_golfers + 1):
        for g2 in range(g1 + 1, n_golfers + 1):
            pair_count += 1
            # Weight by how constraining this pair is
            # More rounds make each pair more constraining
            constraint_strength = min(n_rounds / (n_golfers - 1), 1.0)
            
            constraint_id = f'pair_constraint_{g1}_{g2}'
            G.add_node(constraint_id, type=1, weight=constraint_strength)
            
            # Connect the two golfers involved in this pairing constraint
            G.add_edge(f'golfer_{g1}', constraint_id, weight=1.0)
            G.add_edge(f'golfer_{g2}', constraint_id, weight=1.0)
    
    # Add conflict edges between golfers for highly constrained instances
    # This captures the difficulty of avoiding repeat pairings
    if n_rounds > n_golfers // n_per_group:  # Tight instances
        for g1 in range(1, n_golfers + 1):
            for g2 in range(g1 + 1, n_golfers + 1):
                # Probability they'll conflict increases with rounds/groups ratio
                conflict_prob = min((n_rounds * n_per_group) / n_golfers, 1.0)
                if conflict_prob > 0.7:  # Only add for significant conflicts
                    G.add_edge(f'golfer_{g1}', f'golfer_{g2}', 
                              weight=conflict_prob)
    
    # Connect golfers to round-group slots via group membership constraints
    for r in range(1, n_rounds + 1):
        for gr in range(1, n_groups + 1):
            slot_id = f'slot_r{r}_g{gr}'
            group_constraint = f'group_size_r{r}_g{gr}'
            
            # Each golfer could potentially be assigned to this slot
            for g in range(1, n_golfers + 1):
                # Weight reflects assignment probability and constraint tightness
                assignment_weight = min(n_per_group / n_golfers, 1.0)
                G.add_edge(f'golfer_{g}', group_constraint, weight=assignment_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]  
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()