#!/usr/bin/env python3
"""
Graph converter for golfers1b problem.
Converter created with subagent_prompt.md v_02

This problem is about the social golfers problem: organizing golfers into groups
across multiple rounds such that no two golfers play together more than once.
Key challenges: Quadratic growth in pair constraints with golfers, dense
constraint structure requiring careful scheduling.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the social golfers problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the complex scheduling constraints and pair conflicts
    - Golfers are decision variables (type 0) with centrality-based weights
    - Create constraint nodes for each structural requirement (type 1)
    - Model the quadratic pair-interaction complexity with conflict edges
    - Weight by constraint tightness and structural complexity
    """
    # Access data directly from json_data dict
    n_groups = json_data.get('n_groups', 1)
    n_per_group = json_data.get('n_per_group', 1)  
    n_rounds = json_data.get('n_rounds', 1)
    
    # Calculate problem parameters
    n_golfers = n_groups * n_per_group
    n_places = n_golfers
    total_pairs = n_golfers * (n_golfers - 1) // 2
    pairs_per_round = n_groups * n_per_group * (n_per_group - 1) // 2
    
    # Create graph
    G = nx.Graph()
    
    # Type 0 nodes: Golfers (decision variables)
    # Weight by expected constraint participation (more central = more constrained)
    for g in range(1, n_golfers + 1):
        # Golfers participate in many constraints, weight by interaction density
        participation_weight = min(1.0, (n_rounds * (n_per_group - 1)) / n_golfers)
        G.add_node(f'golfer_{g}', type=0, weight=participation_weight)
    
    # Type 1 nodes: Structural constraints
    
    # Round-based alldifferent constraints (each round must have distinct golfers)
    for r in range(1, n_rounds + 1):
        # All golfers must be assigned exactly once per round
        scope_size = n_golfers  # All golfers involved
        constraint_weight = min(1.0, math.log(scope_size + 1) / math.log(n_golfers + 1))
        G.add_node(f'round_alldiff_{r}', type=1, weight=constraint_weight)
    
    # Group ordering constraints within each round (symmetry breaking)
    for r in range(1, n_rounds + 1):
        for p in range(1, n_places + 1):
            if p % n_per_group != 0:  # Not last in group
                ordering_complexity = (n_per_group - 1) / n_per_group
                G.add_node(f'order_r{r}_p{p}', type=1, weight=ordering_complexity)
    
    # Group-to-group ordering constraints (more symmetry breaking)
    for r in range(1, n_rounds + 1):
        for g in range(n_groups - 1):
            inter_group_weight = 0.8  # High importance for structure
            G.add_node(f'group_order_r{r}_g{g}', type=1, weight=inter_group_weight)
    
    # Pair constraints (most critical - each pair plays at most once)
    pair_constraint_weight = min(1.0, math.log(total_pairs + 1) / 10.0)
    for i in range(1, n_golfers + 1):
        for j in range(i + 1, n_golfers + 1):
            G.add_node(f'pair_{i}_{j}', type=1, weight=pair_constraint_weight)
    
    # Bipartite edges: Connect golfers to constraints they participate in
    
    # Golfers to round constraints
    for r in range(1, n_rounds + 1):
        for g in range(1, n_golfers + 1):
            G.add_edge(f'golfer_{g}', f'round_alldiff_{r}', weight=1.0/n_rounds)
    
    # Golfers to ordering constraints  
    for r in range(1, n_rounds + 1):
        for p in range(1, n_places + 1):
            if p % n_per_group != 0:
                # Each golfer could be assigned to this position
                for g in range(1, n_golfers + 1):
                    position_weight = 0.3  # Lower weight for structural constraints
                    G.add_edge(f'golfer_{g}', f'order_r{r}_p{p}', weight=position_weight)
    
    # Golfers to group ordering constraints
    for r in range(1, n_rounds + 1):
        for group_idx in range(n_groups - 1):
            for g in range(1, n_golfers + 1):
                group_order_weight = 0.4
                G.add_edge(f'golfer_{g}', f'group_order_r{r}_g{group_idx}', weight=group_order_weight)
    
    # Golfers to pair constraints (most important edges)
    for i in range(1, n_golfers + 1):
        for j in range(i + 1, n_golfers + 1):
            # High weight because pair constraints are the core difficulty
            pair_edge_weight = min(1.0, 0.9 * n_rounds / (n_rounds + 1))
            G.add_edge(f'golfer_{i}', f'pair_{i}_{j}', weight=pair_edge_weight)
            G.add_edge(f'golfer_{j}', f'pair_{i}_{j}', weight=pair_edge_weight)
    
    # Add conflict edges between golfers for oversubscribed scenarios
    if pairs_per_round > 0 and n_rounds > 1:
        # Calculate constraint density
        max_possible_pairs = total_pairs
        required_pair_meetings = pairs_per_round * n_rounds
        
        if required_pair_meetings > max_possible_pairs:
            # Oversubscribed - add conflict edges between frequently interacting golfers
            conflict_threshold = 0.7
            for i in range(1, n_golfers + 1):
                for j in range(i + 1, n_golfers + 1):
                    # Golfers that are likely to be in conflict due to scheduling pressure
                    expected_meetings = n_rounds / (n_golfers - 1) * (n_per_group - 1)
                    if expected_meetings > conflict_threshold:
                        conflict_weight = min(1.0, expected_meetings / n_rounds)
                        G.add_edge(f'golfer_{i}', f'golfer_{j}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()