#!/usr/bin/env python3
"""
Graph converter for golfers2 problem.
# Converter created with subagent_prompt.md v_02

This problem is about the Social Golfers Problem: arranging golfers into groups 
across multiple rounds such that no two golfers play together more than once.
Key challenges: combinatorial explosion, constraint propagation complexity, 
symmetry breaking, and the balance between rounds, groups, and golfer pairs.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Social Golfers Problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the constraint structure
    - Golfer nodes (type 0): decision variables, weighted by centrality in constraint network
    - Constraint nodes (type 1): different types of constraints with varying tightness
    - Resource nodes (type 2): rounds and groups as shared resources
    
    Key entities:
    - Golfers: The main decision variables
    - Rounds: Time-based resources (limited slots)
    - Groups: Space-based resources within each round
    - Constraints: All-different (within round), ordering (symmetry), pairing limits
    
    What makes instances hard:
    - High ratio of rounds to theoretical maximum
    - Large group sizes (more pairwise constraints)
    - Constraint interaction density
    """
    
    # Extract problem parameters
    n_groups = json_data.get('n_groups', 2)
    n_per_group = json_data.get('n_per_group', 2)
    n_rounds = json_data.get('n_rounds', 3)
    
    n_golfers = n_groups * n_per_group
    total_pairs = n_golfers * (n_golfers - 1) // 2
    pairs_per_round = n_groups * (n_per_group * (n_per_group - 1) // 2)
    max_possible_rounds = total_pairs // pairs_per_round if pairs_per_round > 0 else 1
    
    G = nx.Graph()
    
    # === GOLFER NODES (Type 0) ===
    # Weight by constraint participation and centrality
    for g in range(1, n_golfers + 1):
        # Golfers participate in: n_rounds placement constraints + pairing constraints
        # Weight higher for golfers with more constraint interactions
        constraint_participation = n_rounds + (n_golfers - 1)  # rounds + potential pairs
        normalized_participation = min(constraint_participation / (n_rounds * n_golfers), 1.0)
        
        # Add non-linear scaling based on problem complexity
        complexity_factor = math.exp(-2.0 * (g - 1) / n_golfers)  # Edge golfers slightly easier
        final_weight = normalized_participation * (0.7 + 0.3 * complexity_factor)
        
        G.add_node(f'golfer_{g}', type=0, weight=final_weight)
    
    # === CONSTRAINT NODES (Type 1) ===
    
    # 1. All-different constraints (one per round)
    for r in range(1, n_rounds + 1):
        # Tightness: all n_golfers must be placed in exactly n_groups * n_per_group slots
        # This is always tight (tightness = 1.0) but varies by round complexity
        round_complexity = 1.0 - math.exp(-3.0 * r / n_rounds)  # Later rounds harder
        G.add_node(f'alldiff_round_{r}', type=1, weight=round_complexity)
    
    # 2. Group ordering constraints (symmetry breaking)
    for r in range(1, n_rounds + 1):
        for g in range(1, n_groups + 1):
            # Weight by group size and position - larger groups have more ordering constraints
            ordering_complexity = (n_per_group - 1) / max(n_per_group, 1)
            # Add positional weight (later groups in later rounds slightly harder)
            position_factor = 1.0 + 0.2 * (g - 1) / max(n_groups - 1, 1)
            final_weight = min(ordering_complexity * position_factor, 1.0)
            G.add_node(f'order_r{r}_g{g}', type=1, weight=final_weight)
    
    # 3. Pairing limit constraints (no pair plays together twice)
    pair_count = 0
    max_pair_weight = 0.0
    for a in range(1, n_golfers + 1):
        for b in range(a + 1, n_golfers + 1):
            # Weight by constraint tightness: how often this pair could potentially meet
            potential_meetings = sum(1 for r in range(n_rounds) 
                                   for g in range(n_groups) 
                                   if pairs_per_round > 0)
            
            # Higher weight for pairs that are more likely to conflict
            conflict_probability = pairs_per_round / max(total_pairs - pairs_per_round, 1)
            pair_weight = min(conflict_probability * 2.0, 1.0)
            max_pair_weight = max(max_pair_weight, pair_weight)
            
            G.add_node(f'pair_limit_{a}_{b}', type=1, weight=pair_weight)
            pair_count += 1
    
    # === RESOURCE NODES (Type 2) ===
    
    # Round resources - weight by utilization and position
    for r in range(1, n_rounds + 1):
        # Weight by how close we are to theoretical maximum
        round_pressure = r / max_possible_rounds if max_possible_rounds > 0 else 0.5
        round_weight = min(round_pressure * 1.5, 1.0)
        G.add_node(f'round_resource_{r}', type=2, weight=round_weight)
    
    # Group resources within rounds
    for r in range(1, n_rounds + 1):
        for g in range(1, n_groups + 1):
            # Weight by group capacity utilization
            utilization = n_per_group / max(n_golfers, 1)
            # Add complexity for managing larger groups
            group_complexity = math.log(n_per_group + 1) / math.log(n_golfers + 1)
            group_weight = min(utilization + group_complexity, 1.0)
            G.add_node(f'group_resource_r{r}_g{g}', type=2, weight=group_weight)
    
    # === EDGES ===
    
    # Golfer participation in round constraints
    for g in range(1, n_golfers + 1):
        for r in range(1, n_rounds + 1):
            # Strong participation - golfer must be in exactly one group this round
            G.add_edge(f'golfer_{g}', f'alldiff_round_{r}', weight=1.0)
            
            # Connection to round resource
            G.add_edge(f'golfer_{g}', f'round_resource_{r}', weight=0.8)
    
    # Golfer participation in group ordering constraints
    for g in range(1, n_golfers + 1):
        for r in range(1, n_rounds + 1):
            for group in range(1, n_groups + 1):
                # Golfer might be in this group (conditional participation)
                participation_strength = 1.0 / n_groups  # Probability of being in this group
                G.add_edge(f'golfer_{g}', f'order_r{r}_g{group}', 
                          weight=participation_strength)
                
                # Connection to group resource
                G.add_edge(f'golfer_{g}', f'group_resource_r{r}_g{group}', 
                          weight=participation_strength)
    
    # Golfer participation in pairing constraints
    for a in range(1, n_golfers + 1):
        for b in range(a + 1, n_golfers + 1):
            # Probability that this pair will be in same group
            pairing_probability = (pairs_per_round * n_rounds) / max(total_pairs, 1)
            edge_weight = min(pairing_probability * 3.0, 1.0)
            
            G.add_edge(f'golfer_{a}', f'pair_limit_{a}_{b}', weight=edge_weight)
            G.add_edge(f'golfer_{b}', f'pair_limit_{a}_{b}', weight=edge_weight)
    
    # Constraint dependencies - rounds depend on each other
    for r1 in range(1, n_rounds):
        for r2 in range(r1 + 1, n_rounds + 1):
            # Later rounds are constrained by earlier round decisions
            dependency_strength = math.exp(-2.0 * (r2 - r1) / n_rounds)
            if dependency_strength > 0.1:  # Only add meaningful dependencies
                G.add_edge(f'alldiff_round_{r1}', f'alldiff_round_{r2}', 
                          weight=dependency_strength)
    
    # Resource capacity constraints between rounds and groups
    for r in range(1, n_rounds + 1):
        for g in range(1, n_groups + 1):
            # Group resources are constrained by round resources
            G.add_edge(f'round_resource_{r}', f'group_resource_r{r}_g{g}', 
                      weight=0.9)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()