#!/usr/bin/env python3
"""
Graph converter for Social Golfers problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling golfers into groups across multiple rounds
such that each pair of golfers plays together at most once.
Key challenges: Complex pairing constraints, combinatorial explosion with size,
balancing group formation with non-repetition constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the social golfers problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with golfers as variables and constraints
    - Golfer nodes (type 0): Each golfer is a decision variable for group assignment
    - Constraint nodes (type 1): Group size constraints, round constraints, pairing constraints
    - Edges represent participation in constraints
    - Weights reflect constraint tightness and golfer interaction complexity
    
    Key insight: Difficulty comes from the exponential growth of pairing constraints
    and the need to balance group formation across multiple rounds.
    """
    # Access data directly from json_data dict
    n_groups = json_data.get('n_groups', 1)
    n_per_group = json_data.get('n_per_group', 1) 
    n_rounds = json_data.get('n_rounds', 1)
    
    n_golfers = n_groups * n_per_group
    
    # Create graph
    G = nx.Graph()
    
    # Golfer nodes (type 0) - each golfer needs to be assigned to groups
    # Weight by interaction potential - golfers with more potential pairings are more constrained
    max_potential_pairs = n_golfers - 1  # Each golfer can potentially pair with all others
    for golfer in range(1, n_golfers + 1):
        # Central golfers (in terms of potential interactions) are more constrained
        interaction_weight = 1.0 if max_potential_pairs == 0 else min(1.0, (n_golfers - 1) / max_potential_pairs)
        G.add_node(f'golfer_{golfer}', type=0, weight=interaction_weight)
    
    # Group size constraints (type 1) - one for each group in each round
    # Weight by relative constraint tightness
    total_slots = n_groups * n_per_group * n_rounds
    for round_num in range(1, n_rounds + 1):
        for group_num in range(1, n_groups + 1):
            # Tightness depends on how restrictive this group size constraint is
            # More rounds = more constrained scheduling
            constraint_tightness = min(1.0, n_rounds / (n_rounds + 1))
            G.add_node(f'group_size_r{round_num}_g{group_num}', type=1, weight=constraint_tightness)
    
    # Round disjointness constraints (type 1) - one for each round
    # These ensure each golfer appears exactly once per round
    for round_num in range(1, n_rounds + 1):
        # Tightness increases with more groups (harder to partition)
        partition_difficulty = min(1.0, n_groups / (n_groups + 1))
        G.add_node(f'round_partition_r{round_num}', type=1, weight=partition_difficulty)
    
    # Pairing constraints (type 1) - one for each pair of golfers
    # These are the most critical constraints - each pair can meet at most once
    total_pairs = (n_golfers * (n_golfers - 1)) // 2
    max_possible_meetings = n_rounds * (n_per_group * (n_per_group - 1)) // 2
    pairing_pressure = min(1.0, total_pairs / max(max_possible_meetings, 1))
    
    for golfer_a in range(1, n_golfers + 1):
        for golfer_b in range(golfer_a + 1, n_golfers + 1):
            # Weight by global pairing pressure - how tight the overall constraint is
            # Use exponential scaling to emphasize high-pressure situations
            pair_weight = min(1.0, math.exp(-2.0 * (1.0 - pairing_pressure)))
            G.add_node(f'pair_{golfer_a}_{golfer_b}', type=1, weight=pair_weight)
    
    # Add edges for constraint participation
    
    # Golfers participate in group size constraints
    for round_num in range(1, n_rounds + 1):
        for group_num in range(1, n_groups + 1):
            constraint_node = f'group_size_r{round_num}_g{group_num}'
            for golfer in range(1, n_golfers + 1):
                # Edge weight represents participation strength
                # All golfers equally participate in group formation
                G.add_edge(f'golfer_{golfer}', constraint_node, weight=1.0)
    
    # Golfers participate in round partition constraints  
    for round_num in range(1, n_rounds + 1):
        constraint_node = f'round_partition_r{round_num}'
        for golfer in range(1, n_golfers + 1):
            # Each golfer must appear exactly once per round
            G.add_edge(f'golfer_{golfer}', constraint_node, weight=1.0)
    
    # Golfers participate in pairing constraints
    for golfer_a in range(1, n_golfers + 1):
        for golfer_b in range(golfer_a + 1, n_golfers + 1):
            pair_constraint = f'pair_{golfer_a}_{golfer_b}'
            # Both golfers in the pair participate in this constraint
            G.add_edge(f'golfer_{golfer_a}', pair_constraint, weight=1.0)
            G.add_edge(f'golfer_{golfer_b}', pair_constraint, weight=1.0)
    
    # Add a global complexity node (type 1) representing overall problem difficulty
    # This captures the exponential nature of the pairing problem
    total_constraints = n_rounds * n_groups + n_rounds + total_pairs
    complexity_ratio = min(1.0, total_constraints / (n_golfers * 10))  # Normalized complexity measure
    complexity_weight = min(1.0, math.exp(-1.0 * (1.0 - complexity_ratio)))
    G.add_node('global_complexity', type=1, weight=complexity_weight)
    
    # Connect high-constraint golfers to global complexity
    for golfer in range(1, n_golfers + 1):
        # Weight by golfer's constraint involvement
        involvement = 1.0  # All golfers equally involved in this problem
        G.add_edge(f'golfer_{golfer}', 'global_complexity', weight=involvement * complexity_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()