#!/usr/bin/env python3
"""
Graph converter for graph-scan-revised problem.
Created using subagent_prompt.md version: v_02

This problem is about coverage/scanning of graph edges with multiple drones.
Each drone must visit and scan certain edges within a makespan constraint.
Key challenges: edge traversal costs, scanning vs traversal time multipliers, 
coordination between drones, ensuring all edges are covered.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the graph scanning problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with edges as variable nodes and constraints
    - Edge nodes (type 0): each edge that needs to be scanned, weighted by scanning difficulty
    - Drone nodes (type 2): resources representing the n drones
    - Constraint nodes (type 1): coverage constraints, timing constraints, capacity constraints
    - Model the scanning coordination problem structure
    
    Note: Limited by JSON conversion - missing succ/reverse arrays from original DZN
    """
    # Access data directly from json_data dict
    n_drones = json_data.get('n', 1)
    scanmultiplier = json_data.get('scanmultiplier', 2)
    edge_lengths = json_data.get('length', [])
    edge_ids = json_data.get('EDGE', [])
    
    n_edges = len(edge_lengths)
    
    G = nx.Graph()
    
    # Calculate statistics for normalization
    if edge_lengths:
        max_length = max(edge_lengths)
        min_length = min(edge_lengths)
        avg_length = sum(edge_lengths) / len(edge_lengths)
        total_work = sum(edge_lengths) * scanmultiplier
    else:
        max_length = min_length = avg_length = total_work = 1
    
    # Edge nodes (type 0) - these are the variables being assigned to drones
    for i, length in enumerate(edge_lengths):
        edge_id = edge_ids[i] if i < len(edge_ids) else i+1
        
        # Weight based on scanning difficulty (longer edges are harder)
        # Use non-linear scaling - exponential for high-cost edges
        if max_length > min_length:
            length_ratio = (length - min_length) / (max_length - min_length)
            # Exponential weighting for scanning difficulty
            scan_difficulty = 0.3 + 0.7 * math.exp(2 * length_ratio) / math.exp(2)
        else:
            scan_difficulty = 0.5
            
        G.add_node(f'edge_{edge_id}', type=0, weight=scan_difficulty)
    
    # Drone resource nodes (type 2) - limited resources
    for d in range(n_drones):
        # Weight by workload capacity (more drones = less individual capacity)
        capacity_weight = 1.0 / (1.0 + math.log(n_drones + 1))
        G.add_node(f'drone_{d}', type=2, weight=capacity_weight)
    
    # Coverage constraint nodes (type 1)
    # Global coverage constraint - all edges must be scanned
    G.add_node('coverage_constraint', type=1, weight=1.0)
    
    # Workload balance constraints for each drone
    expected_workload = total_work / n_drones if n_drones > 0 else total_work
    for d in range(n_drones):
        # Weight by how critical load balancing is (more critical with more drones)
        balance_criticality = math.log(n_drones + 1) / math.log(10)
        G.add_node(f'workload_constraint_{d}', type=1, weight=balance_criticality)
    
    # Makespan constraint - timing constraint
    makespan_tightness = min(1.0, total_work / (n_drones * avg_length * 10))  # Rough estimate
    G.add_node('makespan_constraint', type=1, weight=makespan_tightness)
    
    # Scanning multiplier constraint nodes - one per edge type
    # Group edges by length ranges for scanning constraint modeling
    length_ranges = 3  # Create 3 ranges: short, medium, long
    if max_length > min_length:
        range_size = (max_length - min_length) / length_ranges
        for r in range(length_ranges):
            range_min = min_length + r * range_size
            range_max = min_length + (r + 1) * range_size
            # Count edges in this range
            edges_in_range = sum(1 for length in edge_lengths 
                               if range_min <= length < range_max or (r == length_ranges-1 and length <= range_max))
            if edges_in_range > 0:
                range_difficulty = (range_min + range_max) / (2 * max_length)
                G.add_node(f'scan_constraint_range_{r}', type=1, weight=range_difficulty)
    
    # Add bipartite edges: edge participation in constraints
    for i, length in enumerate(edge_lengths):
        edge_id = edge_ids[i] if i < len(edge_ids) else i+1
        edge_node = f'edge_{edge_id}'
        
        # Every edge participates in global coverage
        G.add_edge(edge_node, 'coverage_constraint', weight=1.0)
        
        # Every edge participates in makespan constraint
        makespan_impact = length * scanmultiplier / total_work if total_work > 0 else 0.5
        G.add_edge(edge_node, 'makespan_constraint', weight=makespan_impact)
        
        # Edge participates in range-based scanning constraints
        if max_length > min_length:
            range_size = (max_length - min_length) / length_ranges
            range_idx = min(length_ranges - 1, int((length - min_length) / range_size))
            range_constraint = f'scan_constraint_range_{range_idx}'
            if G.has_node(range_constraint):
                G.add_edge(edge_node, range_constraint, weight=0.8)
    
    # Add drone-constraint edges
    for d in range(n_drones):
        drone_node = f'drone_{d}'
        
        # Drone participates in its workload constraint
        G.add_edge(drone_node, f'workload_constraint_{d}', weight=1.0)
        
        # Drone participates in global makespan constraint
        makespan_participation = 1.0 / n_drones
        G.add_edge(drone_node, 'makespan_constraint', weight=makespan_participation)
    
    # Add resource competition edges between drones and high-cost edges
    # Identify high-cost edges (top 30%)
    if edge_lengths:
        cost_threshold = sorted(edge_lengths, reverse=True)[min(len(edge_lengths)-1, len(edge_lengths)//3)]
        for i, length in enumerate(edge_lengths):
            if length >= cost_threshold:
                edge_id = edge_ids[i] if i < len(edge_ids) else i+1
                edge_node = f'edge_{edge_id}'
                
                # High-cost edges compete for drone resources
                for d in range(n_drones):
                    drone_node = f'drone_{d}'
                    competition_weight = length / max_length if max_length > 0 else 0.5
                    G.add_edge(edge_node, drone_node, weight=competition_weight)
    
    # Add conflict edges between similar-cost edges (they compete for the same drone type)
    if len(edge_lengths) > 1:
        sorted_edges = sorted(enumerate(edge_lengths), key=lambda x: x[1], reverse=True)
        # Add conflicts between top 50% highest cost edges
        high_cost_edges = sorted_edges[:len(sorted_edges)//2]
        
        for i in range(len(high_cost_edges)):
            for j in range(i+1, min(i+5, len(high_cost_edges))):  # Limit conflicts to nearby costs
                idx1, cost1 = high_cost_edges[i]
                idx2, cost2 = high_cost_edges[j]
                
                edge1_id = edge_ids[idx1] if idx1 < len(edge_ids) else idx1+1
                edge2_id = edge_ids[idx2] if idx2 < len(edge_ids) else idx2+1
                
                # Conflict strength based on cost similarity and drone capacity
                cost_similarity = 1.0 - abs(cost1 - cost2) / max_length if max_length > 0 else 0.5
                drone_scarcity = math.log(max(1, len(edge_lengths) / n_drones)) / math.log(10)
                conflict_weight = cost_similarity * drone_scarcity * 0.3  # Scale down more aggressively
                
                # Ensure weight stays within bounds
                conflict_weight = min(conflict_weight, 1.0)
                
                if conflict_weight > 0.1:  # Only add meaningful conflicts
                    G.add_edge(f'edge_{edge1_id}', f'edge_{edge2_id}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()