#!/usr/bin/env python3
"""
Graph converter for handball_1 problem.
Created using subagent_prompt.md version: v_02

This problem is about handball league scheduling with 14 teams in 2 divisions.
Key challenges: venue unavailability constraints, derby match requirements, balanced scheduling,
home/away patterns, complementary schedules, and break minimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the handball scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model teams as variables, constraints as explicit nodes, and scheduling conflicts.
    - Teams (14) are variable nodes that need scheduling
    - Divisions create grouping constraints
    - Venue unavailability creates resource constraints
    - Derby matches create special timing constraints
    - Home/away balance creates period-based constraints
    """
    # Extract data
    group1 = json_data.get('group1', [])
    group2 = json_data.get('group2', [])
    nohome = json_data.get('nohome', [])
    derby_period = json_data.get('derby_period', [])
    
    # Constants from the model
    num_teams = 14
    num_periods = 20
    num_total_periods = 33  # including mirror schedule
    
    G = nx.Graph()
    
    # Team nodes (Type 0 - variable-like)
    # Weight by division centrality and venue constraints
    for team in range(1, num_teams + 1):
        # Calculate venue unavailability pressure
        team_idx = team - 1
        start_idx = team_idx * num_total_periods
        end_idx = start_idx + num_total_periods
        venue_constraints = sum(nohome[start_idx:end_idx]) if start_idx < len(nohome) else 0
        
        # Central teams in divisions have more scheduling constraints
        if team in group1:
            div_size = len(group1)
            pos_in_div = group1.index(team)
        elif team in group2:
            div_size = len(group2)
            pos_in_div = group2.index(team)
        else:
            div_size = 7
            pos_in_div = 3
        
        # Central teams are more constrained
        centrality = 1.0 - abs(pos_in_div - div_size//2) / (div_size//2 + 1)
        
        # Combine venue pressure and centrality with non-linear weighting
        venue_pressure = venue_constraints / num_total_periods if num_total_periods > 0 else 0
        weight = 0.3 + 0.4 * centrality + 0.3 * venue_pressure
        
        G.add_node(f'team_{team}', type=0, weight=min(weight, 1.0))
    
    # Division constraint nodes (Type 1)
    # Each division has internal tournament constraints
    for div_id, group in enumerate([group1, group2], 1):
        if group:
            # Weight by division size and internal connectivity
            div_tightness = len(group) / 7.0  # normalized division size
            G.add_node(f'division_{div_id}', type=1, weight=div_tightness)
            
            # Connect all teams in division to their division constraint
            for team in group:
                G.add_edge(f'team_{team}', f'division_{div_id}', weight=0.8)
    
    # Period constraint nodes (Type 1)
    # Each period has capacity and scheduling constraints
    for period in range(1, num_periods + 1):
        # Derby periods are more constrained
        is_derby_period = period in derby_period
        derby_weight = 0.9 if is_derby_period else 0.6
        
        # Early and late periods in season are more constrained
        season_position = abs(period - num_periods/2) / (num_periods/2)
        period_weight = 0.4 + 0.3 * derby_weight + 0.3 * season_position
        
        G.add_node(f'period_{period}', type=1, weight=min(period_weight, 1.0))
    
    # Venue availability constraint nodes (Type 1)
    # Create constraints for venue unavailability
    venue_constraint_count = 0
    for team in range(1, num_teams + 1):
        team_idx = team - 1
        start_idx = team_idx * num_total_periods
        end_idx = start_idx + num_total_periods
        
        if start_idx < len(nohome):
            team_nohome = nohome[start_idx:end_idx]
            unavailable_periods = [i for i, val in enumerate(team_nohome) if val == 1]
            
            for period_idx in unavailable_periods:
                venue_constraint_count += 1
                # Weight by scarcity - more unavailable periods = tighter constraint
                scarcity = len(unavailable_periods) / num_total_periods
                constraint_weight = 0.5 + 0.5 * scarcity
                
                constraint_id = f'venue_constraint_{venue_constraint_count}'
                G.add_node(constraint_id, type=1, weight=min(constraint_weight, 1.0))
                
                # Connect team to venue constraint
                G.add_edge(f'team_{team}', constraint_id, weight=0.9)
    
    # Derby match constraint nodes (Type 1)
    # Derby matches have special timing requirements
    for i, period in enumerate(derby_period):
        derby_tightness = 0.8 + 0.2 * (i / max(len(derby_period) - 1, 1))  # Later derbies more constrained
        G.add_node(f'derby_{i}', type=1, weight=derby_tightness)
        
        # Connect period to derby constraint
        if period <= num_periods:
            G.add_edge(f'period_{period}', f'derby_{i}', weight=0.9)
    
    # Home/Away balance constraints (Type 1)
    # Each team needs balanced home/away games
    for team in range(1, num_teams + 1):
        balance_weight = 0.7  # Moderate constraint tightness
        G.add_node(f'balance_{team}', type=1, weight=balance_weight)
        G.add_edge(f'team_{team}', f'balance_{team}', weight=0.8)
    
    # Cross-division scheduling constraints (Type 1)
    # Teams from different divisions play in second tournament
    if group1 and group2:
        cross_div_weight = 0.85  # High tightness for cross-division matches
        G.add_node('cross_division', type=1, weight=cross_div_weight)
        
        # Connect all teams to cross-division constraint
        for team in range(1, num_teams + 1):
            G.add_edge(f'team_{team}', 'cross_division', weight=0.7)
    
    # Add team-team conflict edges for same-division scheduling conflicts
    # Teams in same division have more complex scheduling relationships
    for group in [group1, group2]:
        for i, team1 in enumerate(group):
            for team2 in group[i+1:]:
                # Conflict intensity based on scheduling complexity
                # More venue constraints = higher conflict
                team1_idx = (team1 - 1) * num_total_periods
                team2_idx = (team2 - 1) * num_total_periods
                
                team1_constraints = sum(nohome[team1_idx:team1_idx + num_total_periods]) if team1_idx < len(nohome) else 0
                team2_constraints = sum(nohome[team2_idx:team2_idx + num_total_periods]) if team2_idx < len(nohome) else 0
                
                conflict_intensity = (team1_constraints + team2_constraints) / (2 * num_total_periods)
                conflict_weight = 0.3 + 0.4 * conflict_intensity
                
                if conflict_weight > 0.4:  # Only add significant conflicts
                    G.add_edge(f'team_{team1}', f'team_{team2}', weight=min(conflict_weight, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()