#!/usr/bin/env python3
"""
Graph converter for hoist-benchmark problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling multiple hoists moving jobs through a sequence of tanks
in a cyclic manufacturing process. Each job must spend a certain minimum and maximum
time in each tank, and hoists must coordinate to avoid collisions and conflicts.

Key challenges: resource contention (hoists), timing constraints (min/max times),
spatial conflicts (hoists at same tank), and cycle optimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the hoist scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the scheduling problem as a bipartite graph
    - Tank positions as variable nodes (type 0) - decision points for timing
    - Hoists as resource nodes (type 2) - limited shared resources
    - Timing constraints as constraint nodes (type 1) - min/max time bounds
    - Travel time constraints as constraint nodes (type 1) - spatial relationships
    - Capacity constraints as constraint nodes (type 1) - resource limits
    
    What makes instances hard:
    - Few hoists relative to jobs (resource scarcity)
    - Tight timing windows (tmin close to tmax)
    - Long travel times between distant tanks
    - High job count relative to capacity
    """
    # Extract basic parameters
    ninner = json_data.get('Ninner', 12)  # number of tanks
    multiplier = json_data.get('Multiplier', 1)  # multiplier for scaling
    n = multiplier * ninner  # total number of treatment positions
    hoists = json_data.get('Hoists', 2)  # number of hoists
    capacity = json_data.get('Capacity', 3)  # hoist capacity
    j = json_data.get('J', 9)  # number of jobs
    tmin = json_data.get('tmin', [])  # minimum processing times
    e_flat = json_data.get('e', [])  # flattened empty travel times
    
    # Reconstruct 2D e array from flat representation
    # e is (Tinner x (Ninner+1)) = (Ninner+1) x (Ninner+1)
    tinner = ninner + 1
    e = []
    if len(e_flat) >= tinner * (ninner + 1):
        for i in range(tinner):
            row = []
            for j_idx in range(ninner + 1):
                idx = i * (ninner + 1) + j_idx
                if idx < len(e_flat):
                    row.append(e_flat[idx])
                else:
                    row.append(0)
            e.append(row)
    
    G = nx.Graph()
    
    # Variable nodes: Treatment positions (type 0)
    # Weight by position criticality - positions with longer minimum times are more constraining
    max_tmin = max(tmin) if tmin else 1
    for i in range(n + 1):  # 0 to N (including start position)
        if i == 0:
            # Start position
            weight = 0.1  # Low weight for start
        else:
            # Regular treatment position
            tank_idx = ((i - 1) % ninner)
            if tank_idx < len(tmin):
                # Use exponential scaling for time criticality
                time_criticality = math.exp(tmin[tank_idx] / max_tmin) / math.e
                weight = min(time_criticality, 1.0)
            else:
                weight = 0.5
        G.add_node(f'pos_{i}', type=0, weight=weight)
    
    # Resource nodes: Hoists (type 2)
    # Weight by scarcity - fewer hoists relative to jobs makes them more critical
    hoist_scarcity = min(1.0, j / (hoists * capacity))
    for h in range(hoists):
        G.add_node(f'hoist_{h}', type=2, weight=hoist_scarcity)
    
    # Constraint nodes (type 1): Various constraint types
    
    # 1. Timing constraints for each position
    for i in range(1, n + 1):
        tank_idx = ((i - 1) % ninner)
        if tank_idx < len(tmin):
            # Assume tmax is roughly 1.5 * tmin when not available
            estimated_tmax = tmin[tank_idx] * 1.5
            # Tightness: smaller window = higher weight
            window_tightness = 1.0 - (estimated_tmax - tmin[tank_idx]) / estimated_tmax
            weight = max(0.3, min(1.0, window_tightness))
        else:
            weight = 0.5
        G.add_node(f'timing_constraint_{i}', type=1, weight=weight)
    
    # 2. Travel time constraints between consecutive positions
    for i in range(1, n + 1):
        # Weight by travel time - longer travels create more constraints
        if e and len(e) > 1:
            # Use average travel time from position i-1 to i
            from_tank = ((i - 2) % ninner) if i > 1 else 0
            to_tank = ((i - 1) % ninner)
            if from_tank < len(e) and to_tank < len(e[from_tank]):
                travel_time = e[from_tank][to_tank]
                max_travel = max(max(row) for row in e) if e else 1
                travel_weight = min(1.0, travel_time / max_travel)
            else:
                travel_weight = 0.5
        else:
            travel_weight = 0.5
        G.add_node(f'travel_constraint_{i}', type=1, weight=travel_weight)
    
    # 3. Hoist capacity constraints
    for h in range(hoists):
        # Weight by utilization pressure
        utilization = j / (hoists * capacity)
        weight = min(1.0, utilization)
        G.add_node(f'capacity_constraint_{h}', type=1, weight=weight)
    
    # 4. Global resource constraint (job count limit)
    job_pressure = j / (hoists * capacity)
    weight = min(1.0, job_pressure)
    G.add_node('global_job_constraint', type=1, weight=weight)
    
    # 5. Hoist conflict constraints (for positions that may have hoist conflicts)
    for i in range(1, n + 1):
        for j in range(i + 1, n + 1):
            # Add constraint for potential hoist conflicts
            # Weight by proximity and timing overlap potential
            pos_distance = abs(i - j)
            conflict_potential = 1.0 / (1.0 + pos_distance / 5.0)  # Exponential decay
            if conflict_potential > 0.3:  # Only add if significant conflict potential
                G.add_node(f'hoist_conflict_{i}_{j}', type=1, weight=conflict_potential)
    
    # Edges: Variable-constraint participation (bipartite structure)
    
    # Position to timing constraints
    for i in range(1, n + 1):
        G.add_edge(f'pos_{i}', f'timing_constraint_{i}', weight=1.0)
    
    # Position to travel constraints
    for i in range(1, n + 1):
        G.add_edge(f'pos_{i}', f'travel_constraint_{i}', weight=0.8)
        if i > 1:
            # Previous position also affects travel constraint
            G.add_edge(f'pos_{i-1}', f'travel_constraint_{i}', weight=0.8)
    
    # Positions to hoist resources (potential assignments)
    for i in range(n + 1):
        for h in range(hoists):
            # Weight by position criticality and hoist utilization
            pos_weight = G.nodes[f'pos_{i}']['weight']
            hoist_weight = G.nodes[f'hoist_{h}']['weight']
            edge_weight = (pos_weight + hoist_weight) / 2.0
            G.add_edge(f'pos_{i}', f'hoist_{h}', weight=edge_weight)
    
    # Hoists to capacity constraints
    for h in range(hoists):
        G.add_edge(f'hoist_{h}', f'capacity_constraint_{h}', weight=1.0)
    
    # All positions to global job constraint
    for i in range(1, n + 1):
        weight = G.nodes[f'pos_{i}']['weight']
        G.add_edge(f'pos_{i}', 'global_job_constraint', weight=weight)
    
    # Positions to hoist conflict constraints
    for i in range(1, n + 1):
        for j in range(i + 1, n + 1):
            conflict_node = f'hoist_conflict_{i}_{j}'
            if conflict_node in G.nodes:
                G.add_edge(f'pos_{i}', conflict_node, weight=0.7)
                G.add_edge(f'pos_{j}', conflict_node, weight=0.7)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()