#!/usr/bin/env python3
"""
Graph converter for HRC (Hospital-Resident-Couples) problem.
Created using subagent_prompt.md version: v_02

This problem is about matching residents (including couples) to hospital positions
while maintaining stability with a target number of blocking pairs.
Key challenges: couples must be matched together, complex blocking pair constraints,
hospital capacities, and preference list compatibility.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the HRC problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the complex matching structure
    - Resident nodes (type 0): singles and couples (couples as unified entities)
    - Hospital nodes (type 2): resources with capacity constraints
    - Constraint nodes (type 1): capacity, blocking pair, and stability constraints
    - Edge weights reflect preference ranks, capacity utilization, and constraint tightness
    """
    
    nres = json_data.get('nres', 0)
    ncoup = json_data.get('ncoup', 0) 
    nhosp = json_data.get('nhosp', 0)
    num_bp = json_data.get('num_bp', 0)
    
    max_rpref_len = json_data.get('max_rpref_len', 0)
    max_hpref_len = json_data.get('max_hpref_len', 0)
    
    rpref_len = json_data.get('rpref_len', [])
    hpref_len = json_data.get('hpref_len', [])
    hosp_cap = json_data.get('hosp_cap', [])
    
    rpref = json_data.get('rpref', [])
    hpref = json_data.get('hpref', [])
    hrank = json_data.get('hrank', [])
    
    G = nx.Graph()
    
    # === NODES ===
    
    # Couple nodes (type 0) - treat each couple as a single decision unit
    max_pref_len = max(rpref_len) if rpref_len else 1
    for c in range(1, ncoup + 1):
        r1_idx = (c * 2 - 1) - 1  # Convert to 0-based indexing
        r2_idx = (c * 2) - 1
        
        # Couple weight based on preference list length (longer = more constrained)
        r1_pref_len = rpref_len[r1_idx] if r1_idx < len(rpref_len) else 0
        r2_pref_len = rpref_len[r2_idx] if r2_idx < len(rpref_len) else 0
        avg_pref_len = (r1_pref_len + r2_pref_len) / 2.0
        
        # Couples are more constrained (both must be placed together)
        constraint_weight = 0.8 + 0.2 * (avg_pref_len / max_pref_len)
        G.add_node(f'couple_{c}', type=0, weight=constraint_weight)
    
    # Single resident nodes (type 0)
    singles_start = 2 * ncoup + 1
    for r in range(singles_start, nres + 1):
        r_idx = r - 1  # Convert to 0-based
        pref_len = rpref_len[r_idx] if r_idx < len(rpref_len) else 0
        
        # Weight by preference list length and position criticality
        if max_pref_len > 0:
            pref_weight = 0.3 + 0.7 * (pref_len / max_pref_len)
        else:
            pref_weight = 0.5
        
        G.add_node(f'single_{r}', type=0, weight=pref_weight)
    
    # Hospital nodes (type 2) - resources with capacity
    max_capacity = max(hosp_cap) if hosp_cap else 1
    total_residents = nres
    for h in range(1, nhosp + 1):
        h_idx = h - 1
        capacity = hosp_cap[h_idx] if h_idx < len(hosp_cap) else 1
        pref_len = hpref_len[h_idx] if h_idx < len(hpref_len) else 0
        
        # Hospital competitiveness: high demand vs capacity
        demand_ratio = pref_len / capacity if capacity > 0 else 1.0
        # Use logarithmic scaling for demand pressure
        competitiveness = min(1.0, 0.3 + 0.7 * math.log(1 + demand_ratio) / math.log(6))
        
        G.add_node(f'hospital_{h}', type=2, weight=competitiveness)
    
    # Constraint nodes (type 1)
    
    # 1. Hospital capacity constraints
    for h in range(1, nhosp + 1):
        h_idx = h - 1
        capacity = hosp_cap[h_idx] if h_idx < len(hosp_cap) else 1
        pref_len = hpref_len[h_idx] if h_idx < len(hpref_len) else 0
        
        # Tightness based on demand vs capacity
        if capacity > 0:
            tightness = min(1.0, pref_len / capacity)
        else:
            tightness = 1.0
        
        G.add_node(f'capacity_constraint_{h}', type=1, weight=tightness)
    
    # 2. Blocking pair constraints (different types with different criticality)
    bp_weight = 0.7 + 0.3 * (num_bp / max(1, nres * 0.1))  # Target blocking pairs
    G.add_node('blocking_pairs_constraint', type=1, weight=bp_weight)
    
    # 3. Couple stability constraints (most critical)
    if ncoup > 0:
        G.add_node('couple_stability_constraint', type=1, weight=0.9)
    
    # === EDGES ===
    
    # Couple preferences to hospitals
    for c in range(1, ncoup + 1):
        r1_idx = (c * 2 - 1) - 1
        r1_pref_len = rpref_len[r1_idx] if r1_idx < len(rpref_len) else 0
        
        for j in range(r1_pref_len):
            # Get hospital preference (1-based indexing in rpref)
            rpref_idx = r1_idx * (max_rpref_len + 1) + j
            if rpref_idx < len(rpref) and rpref[rpref_idx] > 0:
                hosp_id = rpref[rpref_idx]
                
                # Edge weight: preference rank (lower rank = higher weight)
                rank_weight = 1.0 - (j / max_rpref_len)
                G.add_edge(f'couple_{c}', f'hospital_{hosp_id}', weight=rank_weight)
    
    # Single resident preferences to hospitals
    for r in range(singles_start, nres + 1):
        r_idx = r - 1
        pref_len = rpref_len[r_idx] if r_idx < len(rpref_len) else 0
        
        for j in range(pref_len):
            rpref_idx = r_idx * (max_rpref_len + 1) + j
            if rpref_idx < len(rpref) and rpref[rpref_idx] > 0:
                hosp_id = rpref[rpref_idx]
                
                rank_weight = 1.0 - (j / max_rpref_len)
                G.add_edge(f'single_{r}', f'hospital_{hosp_id}', weight=rank_weight)
    
    # Hospital capacity constraint edges
    for h in range(1, nhosp + 1):
        # Connect hospital to its capacity constraint
        G.add_edge(f'hospital_{h}', f'capacity_constraint_{h}', weight=1.0)
        
        # Connect residents that prefer this hospital to capacity constraint
        h_idx = h - 1
        pref_len = hpref_len[h_idx] if h_idx < len(hpref_len) else 0
        
        for j in range(pref_len):
            hpref_idx = h_idx * max_hpref_len + j
            if hpref_idx < len(hpref) and hpref[hpref_idx] > 0:
                res_id = hpref[hpref_idx]
                
                # Determine if resident is in a couple or single
                if res_id <= 2 * ncoup:  # Part of a couple
                    couple_id = (res_id + 1) // 2
                    if G.has_node(f'couple_{couple_id}'):
                        # Hospital rank weight (exponential decay for lower preferences)
                        hosp_rank_weight = math.exp(-2.0 * j / max_hpref_len)
                        G.add_edge(f'couple_{couple_id}', f'capacity_constraint_{h}', 
                                 weight=hosp_rank_weight)
                else:  # Single resident
                    if G.has_node(f'single_{res_id}'):
                        hosp_rank_weight = math.exp(-2.0 * j / max_hpref_len)
                        G.add_edge(f'single_{res_id}', f'capacity_constraint_{h}',
                                 weight=hosp_rank_weight)
    
    # Blocking pair constraint edges
    # Connect all residents to blocking pairs constraint (everyone can create blocking pairs)
    bp_edge_weight = 0.5 + 0.5 * (num_bp / max(1, nres * 0.1))
    
    for c in range(1, ncoup + 1):
        if G.has_node(f'couple_{c}'):
            G.add_edge(f'couple_{c}', 'blocking_pairs_constraint', weight=bp_edge_weight)
    
    for r in range(singles_start, nres + 1):
        if G.has_node(f'single_{r}'):
            G.add_edge(f'single_{r}', 'blocking_pairs_constraint', weight=bp_edge_weight)
    
    # Couple stability constraint edges (if couples exist)
    if ncoup > 0:
        for c in range(1, ncoup + 1):
            if G.has_node(f'couple_{c}'):
                # Couples face higher stability requirements
                stability_weight = 0.9
                G.add_edge(f'couple_{c}', 'couple_stability_constraint', weight=stability_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()