#!/usr/bin/env python3
"""
Graph converter for Knights Tour problem.
Created using subagent_prompt.md version: v_02

This problem is about finding a cyclic knight's tour on an n×n chessboard.
Key challenges: Knight movement constraints, cycle closure, path length constraints.
The difficulty increases with board size and tour length requirements.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the knights tour problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: 
    - Board positions as variable nodes (type 0) - where knights can be placed
    - Path constraints as constraint nodes (type 1) - enforce valid knight moves
    - Position constraints as constraint nodes (type 1) - ensure unique visits
    - Edge weights based on movement difficulty and position centrality
    - Tour length and board size affect constraint tightness
    """
    n = json_data.get('n', 8)  # Board size
    m = json_data.get('m', 4)  # Tour length
    
    G = nx.Graph()
    
    # Calculate problem complexity metrics
    total_squares = n * n
    tour_ratio = m / total_squares  # How much of the board is covered
    
    # Variable nodes: Board positions (type 0)
    # Weight by centrality and accessibility (corner/edge positions are harder)
    for row in range(1, n + 1):
        for col in range(1, n + 1):
            # Distance from center (normalized)
            center = (n + 1) / 2
            dist_from_center = abs(row - center) + abs(col - center)
            max_dist = 2 * (n - center)
            
            # Corner and edge penalty (knights have fewer moves from edges)
            edge_penalty = 0
            if row == 1 or row == n or col == 1 or col == n:
                edge_penalty = 0.3
            if (row <= 2 or row >= n-1) and (col <= 2 or col >= n-1):
                edge_penalty = 0.5  # Corner positions are hardest
            
            # Centrality weight (central positions are easier to reach/leave)
            centrality = 1.0 - (dist_from_center / max_dist) if max_dist > 0 else 0.5
            
            # Final weight combines accessibility and tour coverage pressure
            base_weight = centrality - edge_penalty
            tour_pressure = math.sqrt(tour_ratio)  # More tour coverage = more pressure
            
            weight = max(0.1, min(1.0, base_weight * (1.0 + tour_pressure)))
            
            G.add_node(f'pos_{row}_{col}', type=0, weight=weight)
    
    # Constraint nodes for move sequences (type 1)
    # Each position in the tour has movement constraints
    for i in range(1, m):  # m-1 moves between m positions
        # Movement constraint tightness based on available options
        # Knights have at most 8 possible moves from any position
        max_moves = 8
        
        # Constraint becomes tighter as tour progresses (fewer free squares)
        progress_factor = i / m
        constraint_tightness = 0.5 + 0.4 * progress_factor
        
        G.add_node(f'move_constraint_{i}', type=1, weight=constraint_tightness)
    
    # All-different constraint for visited squares (type 1)
    # Tightness increases with tour length relative to board size
    alldiff_tightness = min(1.0, tour_ratio * 1.5)  # More coverage = tighter constraint
    G.add_node('alldiff_constraint', type=1, weight=alldiff_tightness)
    
    # Fixed position constraints (type 1) - these are very tight
    G.add_node('start_constraint', type=1, weight=0.9)  # r[1]=1, c[1]=1
    G.add_node('second_move_constraint', type=1, weight=0.9)  # r[2]=2, c[2]=3  
    G.add_node('end_constraint', type=1, weight=0.9)  # r[m]=3, c[m]=2
    
    # Cycle closure constraint (type 1) - very tight for longer tours
    cycle_tightness = 0.7 + 0.3 * math.sqrt(tour_ratio)
    G.add_node('cycle_constraint', type=1, weight=cycle_tightness)
    
    # Edges: Position participation in constraints
    
    # All positions participate in the all-different constraint
    for row in range(1, n + 1):
        for col in range(1, n + 1):
            pos_node = f'pos_{row}_{col}'
            # Weight by likelihood of being in the tour
            participation_weight = 0.3 + 0.4 * tour_ratio
            G.add_edge(pos_node, 'alldiff_constraint', weight=participation_weight)
    
    # Fixed positions have strong connections to their constraints
    G.add_edge('pos_1_1', 'start_constraint', weight=1.0)
    G.add_edge('pos_2_3', 'second_move_constraint', weight=1.0)
    G.add_edge('pos_3_2', 'end_constraint', weight=1.0)
    
    # Movement constraints connect to reachable positions
    # Each move constraint connects positions that could be consecutive
    knight_moves = [(-2, -1), (-2, 1), (-1, -2), (-1, 2), (1, -2), (1, 2), (2, -1), (2, 1)]
    
    for i in range(1, m):
        move_constraint = f'move_constraint_{i}'
        
        # Connect all valid position pairs for this move
        for row in range(1, n + 1):
            for col in range(1, n + 1):
                pos1 = f'pos_{row}_{col}'
                
                # Find all positions reachable by knight move
                reachable_count = 0
                for dr, dc in knight_moves:
                    new_row, new_col = row + dr, col + dc
                    if 1 <= new_row <= n and 1 <= new_col <= n:
                        pos2 = f'pos_{new_row}_{new_col}'
                        reachable_count += 1
                        
                        # Weight by move difficulty (fewer options = harder)
                        move_weight = 0.4 + 0.6 * (reachable_count / 8.0)
                        
                        G.add_edge(pos1, move_constraint, weight=move_weight * 0.7)
                        G.add_edge(pos2, move_constraint, weight=move_weight * 0.7)
    
    # Cycle constraint connects start and end positions
    G.add_edge('pos_1_1', 'cycle_constraint', weight=0.8)
    G.add_edge('pos_3_2', 'cycle_constraint', weight=0.8)
    
    # Add conflict edges between positions that are too far apart for knight moves
    # (Only for very constrained instances where this matters)
    if tour_ratio > 0.6:  # Only for long tours
        for row1 in range(1, n + 1):
            for col1 in range(1, n + 1):
                for row2 in range(row1, n + 1):
                    for col2 in range(col1 + 1, n + 1):
                        # Check if positions are NOT knight-move reachable
                        dr, dc = abs(row1 - row2), abs(col1 - col2)
                        is_knight_move = (dr == 2 and dc == 1) or (dr == 1 and dc == 2)
                        
                        if not is_knight_move and (dr > 3 or dc > 3):  # Far apart
                            # These positions create path length pressure
                            distance = math.sqrt(dr*dr + dc*dc)
                            max_distance = math.sqrt(2) * n
                            conflict_weight = min(0.3, distance / max_distance)
                            
                            G.add_edge(f'pos_{row1}_{col1}', f'pos_{row2}_{col2}', 
                                     weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()