#!/usr/bin/env python3
"""
Graph converter for Langford problem.
Created using subagent_prompt.md version: v_02

This problem is about arranging k sets of numbers 1 to n so that each appearance 
of number m is m numbers apart from the previous appearance.

Key challenges: 
- Constraint satisfaction with strict distance requirements
- Dual model with channeling constraints between position and number variables  
- Strong interactions between position constraints and alldifferent constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Langford problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the intricate constraint structure of Langford's problem
    - Position variables for each (number, set) pair
    - Distance constraints between consecutive appearances of same number
    - Alldifferent constraints on positions and number assignments
    - Channeling constraints between dual models
    - Higher weights for numbers with stricter distance requirements
    """
    k = json_data.get('k', 2)  # number of sets
    n = json_data.get('n', 3)  # numbers 1..n
    
    G = nx.Graph()
    
    # === Position Variables (Type 0) ===
    # Each (number, set) pair has a position variable
    # Weight by constraint difficulty - larger numbers have stricter distance requirements
    for i in range(1, n+1):  # numbers 1..n
        for j in range(1, k+1):  # sets 1..k
            var_id = f'pos_{i}_{j}'
            # Larger numbers are more constrained (distance = i+1)
            constraint_strength = i / n
            G.add_node(var_id, type=0, weight=constraint_strength)
    
    # === Position Assignment Variables (Type 0) ===
    # Dual model: what (number,set) pair is at each position
    total_positions = n * k
    for p in range(1, total_positions+1):
        var_id = f'num_at_{p}'
        # Central positions are more constrained due to distance requirements
        centrality = 1.0 - abs(p - total_positions/2) / (total_positions/2)
        G.add_node(var_id, type=0, weight=centrality * 0.7)
    
    # === Distance Constraints (Type 1) ===
    # For each number i, consecutive sets must be i+1 positions apart
    for i in range(1, n+1):
        for j in range(1, k):  # j to j+1
            constraint_id = f'distance_{i}_{j}'
            # Larger distance requirements are harder to satisfy
            difficulty = (i + 1) / (total_positions / 2)  # normalized by max possible distance
            G.add_node(constraint_id, type=1, weight=min(difficulty, 1.0))
            
            # Connect to the position variables this constraint affects
            pos_var1 = f'pos_{i}_{j}'
            pos_var2 = f'pos_{i}_{j+1}'
            
            # Strong connections as these variables directly determine constraint satisfaction
            G.add_edge(constraint_id, pos_var1, weight=0.9)
            G.add_edge(constraint_id, pos_var2, weight=0.9)
    
    # === Alldifferent Constraints (Type 1) ===
    # All position variables must be different
    pos_alldiff_id = 'alldiff_positions'
    # Tightness depends on how constrained the space is
    tightness = 1.0 - (total_positions - (n * k)) / total_positions if total_positions > n * k else 0.8
    G.add_node(pos_alldiff_id, type=1, weight=tightness)
    
    # All number assignment variables must be different
    num_alldiff_id = 'alldiff_numbers'
    G.add_node(num_alldiff_id, type=1, weight=tightness)
    
    # Connect alldifferent constraints to all relevant variables
    for i in range(1, n+1):
        for j in range(1, k+1):
            pos_var = f'pos_{i}_{j}'
            # Weight by how much this variable contributes to constraint difficulty
            var_contribution = (i / n) * 0.6  # higher numbers contribute more to difficulty
            G.add_edge(pos_alldiff_id, pos_var, weight=var_contribution)
    
    for p in range(1, total_positions+1):
        num_var = f'num_at_{p}'
        G.add_edge(num_alldiff_id, num_var, weight=0.5)
    
    # === Channeling Constraints (Type 1) ===
    # Connect primal and dual models: Pos[ns] = p <-> Num[p] = ns
    for i in range(1, n+1):
        for j in range(1, k+1):
            for p in range(1, total_positions+1):
                channel_id = f'channel_{i}_{j}_{p}'
                # These are critical for model consistency
                G.add_node(channel_id, type=1, weight=0.8)
                
                pos_var = f'pos_{i}_{j}'
                num_var = f'num_at_{p}'
                
                # Strong bidirectional connection
                G.add_edge(channel_id, pos_var, weight=0.7)
                G.add_edge(channel_id, num_var, weight=0.7)
    
    # === Conflict Edges Between Variables ===
    # Add conflict edges between position variables that compete for similar positions
    # especially for numbers with large distance requirements
    for i in range(1, n+1):
        if i >= n/2:  # Focus on larger numbers which are more constrained
            for j1 in range(1, k+1):
                for j2 in range(j1+1, k+1):
                    pos_var1 = f'pos_{i}_{j1}'
                    pos_var2 = f'pos_{i}_{j2}'
                    # Conflict strength based on required distance vs available space
                    required_distance = i + 1
                    conflict_strength = required_distance / (total_positions / 2)
                    if conflict_strength > 0.3:  # Only add meaningful conflicts
                        G.add_edge(pos_var1, pos_var2, weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()