#!/usr/bin/env python3
"""
Graph converter for Latin Squares FD problem.
Created using subagent_prompt.md version: v_02

This problem is about placing values 1..n in an n×n grid such that each row and 
column contains each value exactly once. The FD version uses 3D boolean variables
x[i,j,k] indicating whether position (i,j) has value k.

Key challenges: 
- Triple coordination between rows, columns, and values
- Quadratic number of all-different constraints
- Heavy constraint propagation requirements
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Latin Squares problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the triple constraint structure
    - Position variables (type 0): Each cell position (i,j) 
    - Value constraint nodes (type 1): Row-value, column-value, and position constraints
    - Resource nodes (type 2): Values 1..n as shared resources
    
    The difficulty comes from the dense constraint interaction where each position
    affects multiple row-value and column-value constraints simultaneously.
    """
    n = json_data.get('n', 3)
    
    G = nx.Graph()
    
    # Position variables (type 0) - decision points for each cell
    # Weight by constraint degree: corner positions are less constrained
    for i in range(1, n+1):
        for j in range(1, n+1):
            # Central positions have more constraint interactions
            centrality = 1.0 - (abs(i - (n+1)/2) + abs(j - (n+1)/2)) / n
            # Normalize to [0,1] with some base weight
            position_weight = 0.3 + 0.7 * centrality
            G.add_node(f'pos_{i}_{j}', type=0, weight=position_weight)
    
    # Row-value constraint nodes (type 1) - each row must contain each value once
    for i in range(1, n+1):
        for k in range(1, n+1):
            # All row-value constraints have same scope (n positions)
            G.add_node(f'row_{i}_val_{k}', type=1, weight=1.0)
    
    # Column-value constraint nodes (type 1) - each column must contain each value once  
    for j in range(1, n+1):
        for k in range(1, n+1):
            # All column-value constraints have same scope (n positions)
            G.add_node(f'col_{j}_val_{k}', type=1, weight=1.0)
    
    # Position constraint nodes (type 1) - each position must have exactly one value
    for i in range(1, n+1):
        for j in range(1, n+1):
            # These constraints have scope n (one for each possible value)
            G.add_node(f'pos_constraint_{i}_{j}', type=1, weight=1.0)
    
    # Value resource nodes (type 2) - values 1..n as shared resources
    for k in range(1, n+1):
        # Each value must be placed exactly n times (once per row and column)
        # Weight by scarcity - all values equally scarce in latin squares
        G.add_node(f'value_{k}', type=2, weight=1.0)
    
    # Bipartite edges: Connect positions to constraints they participate in
    for i in range(1, n+1):
        for j in range(1, n+1):
            pos_node = f'pos_{i}_{j}'
            
            # Connect to position constraint (must choose exactly one value)
            G.add_edge(pos_node, f'pos_constraint_{i}_{j}', weight=1.0)
            
            # Connect to all row-value constraints for this row
            for k in range(1, n+1):
                # Strength based on how constrained this value placement is
                # Use exponential decay based on remaining positions
                constraint_tightness = 1.0 - math.exp(-2.0 / n)
                G.add_edge(pos_node, f'row_{i}_val_{k}', weight=constraint_tightness)
            
            # Connect to all column-value constraints for this column
            for k in range(1, n+1):
                constraint_tightness = 1.0 - math.exp(-2.0 / n)
                G.add_edge(pos_node, f'col_{j}_val_{k}', weight=constraint_tightness)
    
    # Connect constraint nodes to value resources
    for k in range(1, n+1):
        value_node = f'value_{k}'
        
        # Connect row-value constraints to the value resource
        for i in range(1, n+1):
            # Weight by resource consumption (1/n of the total value usage)
            consumption = 1.0 / n
            G.add_edge(f'row_{i}_val_{k}', value_node, weight=consumption)
        
        # Connect column-value constraints to the value resource  
        for j in range(1, n+1):
            consumption = 1.0 / n
            G.add_edge(f'col_{j}_val_{k}', value_node, weight=consumption)
    
    # Add conflict edges between overlapping constraints (non-bipartite)
    # Row-column intersections create the hardest conflicts
    for i in range(1, n+1):
        for j in range(1, n+1):
            for k in range(1, n+1):
                # Row and column constraints for same value at intersection
                row_constraint = f'row_{i}_val_{k}'
                col_constraint = f'col_{j}_val_{k}'
                
                # Conflict strength based on constraint intersection complexity
                # Higher for larger n due to more propagation
                conflict_weight = min(1.0, math.log(n + 1) / math.log(26))  # Normalize to [0,1]
                G.add_edge(row_constraint, col_constraint, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()