#!/usr/bin/env python3
"""
Graph converter for latin-squares-lp problem.
Converter created with subagent_prompt.md v_02

This problem is about placing numbers 1..n in an n×n grid such that each number
appears exactly once in each row and column (Latin square constraints).
Key challenges: combinatorial explosion with size, constraint interactions
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Latin squares problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with cells and constraint nodes
    - Cell nodes (type 0): Each position (i,j) in the grid
    - Constraint nodes (type 1): Row constraints, column constraints, all-different per cell
    - Constraint tightness increases with n (more values competing for positions)
    - Central cells participate in more constraint interactions
    """
    n = json_data.get('n', 3)
    
    G = nx.Graph()
    
    # Add cell nodes (type 0) - each position in the n×n grid
    for i in range(n):
        for j in range(n):
            # Central positions are more constrained (participate in more conflicts)
            # Weight by inverse distance to center, normalized
            center = (n - 1) / 2.0
            dist_to_center = math.sqrt((i - center)**2 + (j - center)**2)
            max_dist = math.sqrt(2) * center if n > 1 else 1.0
            centrality = 1.0 - (dist_to_center / max_dist)
            
            # Add non-linear scaling based on problem size
            size_factor = math.log(n) / math.log(25) if n > 1 else 0.5
            weight = centrality * 0.7 + size_factor * 0.3
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=min(weight, 1.0))
    
    # Add row constraint nodes (type 1)
    for i in range(n):
        # Row constraints become tighter with larger n (more values to place)
        tightness = 1.0 - math.exp(-n/10.0)  # Non-linear scaling
        G.add_node(f'row_{i}', type=1, weight=tightness)
    
    # Add column constraint nodes (type 1)  
    for j in range(n):
        # Column constraints have same tightness as rows
        tightness = 1.0 - math.exp(-n/10.0)
        G.add_node(f'col_{j}', type=1, weight=tightness)
    
    # Add value placement constraint nodes (type 1)
    # Each value k must appear exactly once in each row and column
    for k in range(1, n+1):
        for i in range(n):
            # Row-value constraints
            scope_weight = math.log(n+1) / math.log(26)  # Logarithmic scaling
            G.add_node(f'row_val_{i}_{k}', type=1, weight=scope_weight)
        
        for j in range(n):
            # Column-value constraints
            G.add_node(f'col_val_{j}_{k}', type=1, weight=scope_weight)
    
    # Add edges: cells to their participating constraints
    for i in range(n):
        for j in range(n):
            cell = f'cell_{i}_{j}'
            
            # Connect to row and column constraints (basic participation)
            G.add_edge(cell, f'row_{i}', weight=1.0)
            G.add_edge(cell, f'col_{j}', weight=1.0)
            
            # Connect to value-specific constraints with value-dependent weights
            for k in range(1, n+1):
                # Weight by how much this value contributes to constraint tightness
                value_weight = k / n  # Higher values create more complex interactions
                G.add_edge(cell, f'row_val_{i}_{k}', weight=value_weight)
                G.add_edge(cell, f'col_val_{j}_{k}', weight=value_weight)
    
    # Add some constraint-constraint edges for complex interactions
    # Connect row and column constraints that share cells
    for i in range(n):
        for j in range(n):
            # Interaction strength based on how central the intersection is
            center = (n - 1) / 2.0
            interaction = 1.0 - abs(i - center)/center * abs(j - center)/center if center > 0 else 0.5
            interaction_weight = interaction * 0.3  # Moderate interaction strength
            G.add_edge(f'row_{i}', f'col_{j}', weight=interaction_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()