#!/usr/bin/env python3
"""
Graph converter for League problem.
Created using subagent_prompt.md version: v_02

This problem is about creating tournament groups for players/teams.
The goal is to balance groups by ranking (close skill levels) while maximizing country diversity.
Key challenges: balancing group sizes, minimizing ranking differences, maximizing country variety
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the league grouping problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with players and constraints
    - Players (type 0) represent decision points for group assignment
    - Group constraints (type 1) enforce size limits and balance
    - Ranking constraints (type 1) enforce skill level balance
    - Country diversity constraints (type 1) promote variety
    """
    # Access data from json_data dict
    n = json_data.get('n', 0)
    n_persons_in_group = json_data.get('n_persons_in_group', 3)
    ranking = json_data.get('ranking', [])
    country = json_data.get('country', [])
    
    # Calculate derived values
    n_groups = (n + n_persons_in_group - 1) // n_persons_in_group
    max_country = max(country) if country else 1
    min_rank = min(ranking) if ranking else 1
    max_rank = max(ranking) if ranking else 1
    rank_range = max_rank - min_rank if max_rank > min_rank else 1
    
    G = nx.Graph()
    
    # Type 0: Player nodes with ranking-based weights
    # Higher-ranked (better) players have higher weights as they're more valuable for balance
    for i in range(n):
        player_rank = ranking[i] if i < len(ranking) else min_rank
        # Invert ranking so better players (lower rank numbers) get higher weights
        rank_weight = 1.0 - (player_rank - min_rank) / max(rank_range, 1)
        G.add_node(f'player_{i}', type=0, weight=rank_weight)
    
    # Type 1: Group size constraint nodes
    # Weight by how restrictive the size constraint is relative to total players
    group_size_tightness = n_persons_in_group / max(n, 1)
    for g in range(n_groups):
        G.add_node(f'group_size_{g}', type=1, weight=group_size_tightness)
    
    # Type 1: Ranking balance constraint nodes for each group
    # Weight by expected difficulty of maintaining ranking balance
    for g in range(n_groups):
        # Groups with more diverse rankings are harder to balance
        expected_diversity = min(1.0, n_persons_in_group / max(rank_range, 1))
        balance_difficulty = 1.0 - expected_diversity
        G.add_node(f'rank_balance_{g}', type=1, weight=balance_difficulty)
    
    # Type 1: Country diversity constraint nodes for each group
    # Weight by how challenging it is to achieve diversity given country distribution
    country_counts = {}
    for c in country:
        country_counts[c] = country_counts.get(c, 0) + 1
    
    # Calculate country imbalance (higher when some countries dominate)
    total_players = len(country)
    country_imbalance = 0.0
    if total_players > 0:
        expected_per_country = total_players / max_country
        for count in country_counts.values():
            country_imbalance += abs(count - expected_per_country) / total_players
        country_imbalance /= max_country
    
    for g in range(n_groups):
        diversity_difficulty = min(1.0, country_imbalance + 0.3)  # Base difficulty
        G.add_node(f'country_diversity_{g}', type=1, weight=diversity_difficulty)
    
    # Type 1: Global constraints
    # Overall tournament balance constraint
    tournament_complexity = min(1.0, (n * max_country * rank_range) / (100.0 * n_groups))
    G.add_node('tournament_balance', type=1, weight=tournament_complexity)
    
    # Bipartite edges: Player participation in constraints
    for i in range(n):
        player_id = f'player_{i}'
        player_rank = ranking[i] if i < len(ranking) else min_rank
        player_country = country[i] if i < len(country) else 1
        
        # Connect to all group size constraints (potential assignment)
        for g in range(n_groups):
            # Weight by assignment preference based on current group filling
            assignment_weight = 0.7  # Base weight for all potential assignments
            G.add_edge(player_id, f'group_size_{g}', weight=assignment_weight)
        
        # Connect to all ranking balance constraints
        for g in range(n_groups):
            # Weight by how much this player affects ranking balance
            rank_impact = (player_rank - min_rank) / max(rank_range, 1)
            balance_weight = 0.5 + 0.5 * rank_impact  # More extreme ranks have higher impact
            G.add_edge(player_id, f'rank_balance_{g}', weight=balance_weight)
        
        # Connect to all country diversity constraints
        for g in range(n_groups):
            # Weight by rarity of player's country
            country_rarity = 1.0 - (country_counts.get(player_country, 1) / total_players)
            diversity_weight = 0.4 + 0.6 * country_rarity  # Rarer countries more valuable
            G.add_edge(player_id, f'country_diversity_{g}', weight=diversity_weight)
        
        # Connect to global tournament balance
        # Weight by player's overall impact on tournament quality
        rank_centrality = 1.0 - abs(player_rank - (min_rank + max_rank) / 2) / max(rank_range / 2, 1)
        global_weight = 0.3 + 0.4 * rank_centrality
        G.add_edge(player_id, 'tournament_balance', weight=global_weight)
    
    # Constraint-to-constraint edges for interdependencies
    for g in range(n_groups):
        # Group size affects ranking balance difficulty
        size_balance_weight = min(1.0, n_persons_in_group / max(rank_range, 1))
        G.add_edge(f'group_size_{g}', f'rank_balance_{g}', weight=size_balance_weight)
        
        # Group size affects country diversity difficulty  
        size_diversity_weight = min(1.0, n_persons_in_group / max(max_country, 1))
        G.add_edge(f'group_size_{g}', f'country_diversity_{g}', weight=size_diversity_weight)
        
        # Ranking balance and country diversity can conflict
        balance_diversity_tension = 0.6  # These goals can be in tension
        G.add_edge(f'rank_balance_{g}', f'country_diversity_{g}', weight=balance_diversity_tension)
        
        # Each group constraint connects to global tournament balance
        global_connection_weight = 0.4
        G.add_edge(f'rank_balance_{g}', 'tournament_balance', weight=global_connection_weight)
        G.add_edge(f'country_diversity_{g}', 'tournament_balance', weight=global_connection_weight)
    
    # Add conflict edges between players from same country for over-represented countries
    over_representation_threshold = total_players / (max_country * 1.5)
    for c, count in country_counts.items():
        if count > over_representation_threshold:
            # Find players from this over-represented country
            same_country_players = [i for i in range(n) 
                                   if i < len(country) and country[i] == c]
            
            # Add conflict edges between them (they compete for diversity slots)
            for i in range(len(same_country_players)):
                for j in range(i + 1, len(same_country_players)):
                    p1 = same_country_players[i]
                    p2 = same_country_players[j]
                    # Weight by how over-represented this country is
                    conflict_strength = min(1.0, count / over_representation_threshold - 1.0)
                    G.add_edge(f'player_{p1}', f'player_{p2}', weight=0.3 + 0.4 * conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()