#!/usr/bin/env python3
"""
Graph converter for league_1 problem.
# Converter created with subagent_prompt.md v_02

This problem is about assigning players to leagues while balancing ranking similarity
and country diversity within each league.
Key challenges: Multi-objective optimization (ranking tightness vs. country diversity),
assignment constraints with variable league sizes.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the league assignment problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model players, leagues, and constraints in a bipartite structure
    - Players (type 0): weighted by ranking diversity impact and country rarity
    - League constraints (type 1): capacity, ranking range, country diversity
    - Assignment edges: player-league compatibility based on ranking and country fit
    """
    # Access data directly from json_data dict
    n = json_data.get('n', 0)
    n_persons_in_league = json_data.get('n_persons_in_league', 3)
    ranking = json_data.get('ranking', [])
    country = json_data.get('country', [])
    
    G = nx.Graph()
    
    if n == 0 or not ranking or not country:
        return G
    
    # Calculate problem statistics for meaningful weights
    n_league = (n + n_persons_in_league - 1) // n_persons_in_league
    min_rank = min(ranking)
    max_rank = max(ranking)
    rank_range = max_rank - min_rank + 1
    max_country = max(country)
    
    # Count ranking and country frequencies for rarity weights
    rank_counts = {}
    country_counts = {}
    for i in range(n):
        r = ranking[i]
        c = country[i]
        rank_counts[r] = rank_counts.get(r, 0) + 1
        country_counts[c] = country_counts.get(c, 0) + 1
    
    # Add player nodes (type 0) with ranking and country rarity weights
    for i in range(n):
        player_rank = ranking[i]
        player_country = country[i]
        
        # Ranking rarity weight (players with rare rankings are more valuable)
        rank_rarity = 1.0 - (rank_counts[player_rank] / n)
        
        # Country rarity weight  
        country_rarity = 1.0 - (country_counts[player_country] / n)
        
        # Ranking extremity (players at extremes are harder to place)
        rank_extremity = abs(player_rank - (min_rank + max_rank) / 2) / (rank_range / 2) if rank_range > 0 else 0.5
        
        # Combined weight emphasizing placement difficulty
        player_weight = 0.4 * rank_rarity + 0.3 * country_rarity + 0.3 * rank_extremity
        
        G.add_node(f'player_{i}', type=0, weight=min(player_weight, 1.0))
    
    # Add league constraint nodes (type 1)
    for league_id in range(n_league):
        # League capacity constraint - tighter for larger leagues due to more complex assignment
        capacity_pressure = min(n_persons_in_league / n, 1.0)
        G.add_node(f'league_capacity_{league_id}', type=1, weight=capacity_pressure)
        
        # Ranking homogeneity constraint - more critical for leagues that must span many ranks
        expected_rank_span = rank_range / n_league if n_league > 0 else rank_range
        ranking_difficulty = min(expected_rank_span / rank_range, 1.0) if rank_range > 0 else 0.5
        G.add_node(f'league_ranking_{league_id}', type=1, weight=ranking_difficulty)
        
        # Country diversity constraint - harder when few countries available
        diversity_requirement = min(max_country / n_persons_in_league, 1.0) if n_persons_in_league > 0 else 0.5
        G.add_node(f'league_diversity_{league_id}', type=1, weight=diversity_requirement)
    
    # Add global constraints
    # Overall ranking distribution constraint
    ranking_distribution_complexity = len(set(ranking)) / rank_range if rank_range > 0 else 0.5
    G.add_node('global_ranking_balance', type=1, weight=ranking_distribution_complexity)
    
    # Country distribution constraint
    country_distribution_complexity = len(set(country)) / max_country if max_country > 0 else 0.5
    G.add_node('global_country_balance', type=1, weight=country_distribution_complexity)
    
    # Add edges: player-constraint participation with compatibility weights
    for i in range(n):
        player_rank = ranking[i]
        player_country = country[i]
        
        # Connect players to league constraints they could participate in
        for league_id in range(n_league):
            # Capacity constraint edge - all players can potentially join any league
            G.add_edge(f'player_{i}', f'league_capacity_{league_id}', weight=0.8)
            
            # Ranking constraint edge - weight by ranking fit
            # Calculate expected ranking range for this league
            league_start_rank = min_rank + (league_id * rank_range) // n_league
            league_end_rank = min_rank + ((league_id + 1) * rank_range) // n_league
            
            # Distance from player rank to league's expected range
            if league_start_rank <= player_rank <= league_end_rank:
                rank_fit = 1.0  # Perfect fit
            else:
                distance = min(abs(player_rank - league_start_rank), abs(player_rank - league_end_rank))
                rank_fit = math.exp(-2.0 * distance / rank_range) if rank_range > 0 else 0.5
            
            G.add_edge(f'player_{i}', f'league_ranking_{league_id}', weight=rank_fit)
            
            # Diversity constraint edge - weight by country contribution to diversity
            # Higher weight if this country is needed for diversity
            diversity_contribution = country_rarity = 1.0 - (country_counts[player_country] / n)
            G.add_edge(f'player_{i}', f'league_diversity_{league_id}', weight=diversity_contribution)
        
        # Connect to global constraints
        # Global ranking balance - weight by ranking frequency impact
        ranking_impact = rank_counts[player_rank] / n
        G.add_edge(f'player_{i}', 'global_ranking_balance', weight=1.0 - ranking_impact)
        
        # Global country balance - weight by country frequency impact  
        country_impact = country_counts[player_country] / n
        G.add_edge(f'player_{i}', 'global_country_balance', weight=1.0 - country_impact)
    
    # Add conflict edges between players with very different rankings
    # Only for extreme ranking differences that would make league assignment difficult
    for i in range(n):
        for j in range(i + 1, min(i + 10, n)):  # Limit conflicts to avoid complete graph
            rank_diff = abs(ranking[i] - ranking[j])
            if rank_diff > rank_range * 0.7:  # Large ranking difference
                # Check if they're likely to compete for league assignments
                same_country = (country[i] == country[j])
                conflict_weight = (rank_diff / rank_range) * (1.2 if same_country else 0.8)
                if conflict_weight > 0.6:
                    G.add_edge(f'player_{i}', f'player_{j}', weight=min(conflict_weight, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()