#!/usr/bin/env python3
"""
Graph converter for Multi-Agent Collective Construction (MACC) problem.
Created using subagent_prompt.md version: v_02

This problem is about agents constructing a 3D building by moving blocks on a grid.
Key challenges: spatial coordination, temporal sequencing, agent collision avoidance, and multi-level construction.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the MACC problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model spatial relationships and construction constraints
    - Grid positions as variable nodes with accessibility weights
    - Agent constraints (collision, movement) as constraint nodes
    - Temporal sequence constraints for construction order
    - Building blocks as targets with construction difficulty weights
    """
    # Extract problem parameters
    A = json_data.get('A', 1)  # Number of agents
    T = json_data.get('T', 1)  # Time horizon
    X = json_data.get('X', 1)  # Width
    Y = json_data.get('Y', 1)  # Depth
    Z = json_data.get('Z', 1)  # Height
    building = json_data.get('building', [])
    
    # Grid size
    grid_size = X * Y
    
    # Convert building array to 2D coordinates with heights
    building_blocks = {}
    for y in range(Y):
        for x in range(X):
            idx = y * X + x
            if idx < len(building) and building[idx] > 0:
                building_blocks[(x, y)] = building[idx]
    
    G = nx.Graph()
    
    # Variable nodes: Grid positions with accessibility-based weights
    max_distance_to_border = max(X//2, Y//2)
    for x in range(X):
        for y in range(Y):
            # Calculate distance to border (accessibility measure)
            dist_to_border = min(x, X-1-x, y, Y-1-y)
            
            # Interior positions are more constrained due to agent movement limitations
            # Use exponential decay for accessibility weight
            accessibility = math.exp(-2.0 * (max_distance_to_border - dist_to_border) / max(max_distance_to_border, 1))
            
            # Building targets are more critical
            if (x, y) in building_blocks:
                # Higher weight for taller building blocks (more construction steps needed)
                height = building_blocks[(x, y)]
                target_weight = min(0.3 + 0.7 * height / Z, 1.0)
            else:
                target_weight = 0.0
            
            # Combine accessibility and target importance
            weight = min(0.5 * accessibility + 0.5 * target_weight, 1.0)
            
            G.add_node(f'pos_{x}_{y}', type=0, weight=weight)
    
    # Constraint nodes: Agent coordination constraints
    
    # 1. Agent capacity constraint (global resource)
    agent_utilization = min(len(building_blocks) / (A * T * 0.5), 1.0)  # Rough estimate of workload
    G.add_node('agent_capacity', type=1, weight=agent_utilization)
    
    # 2. Time horizon constraints - one per time step
    total_work = sum(building_blocks.values())
    for t in range(T):
        # Earlier time steps are more constrained due to sequential dependencies
        time_pressure = 1.0 - (t / max(T-1, 1))
        # Weight by remaining work ratio
        remaining_work_ratio = max(0.0, (total_work - t * total_work / T) / max(total_work, 1))
        time_weight = 0.3 + 0.7 * time_pressure * remaining_work_ratio
        G.add_node(f'time_{t}', type=1, weight=min(time_weight, 1.0))
    
    # 3. Building sequence constraints - for positions requiring construction
    for (x, y), height in building_blocks.items():
        # Weight by construction complexity (height and coordination needed)
        complexity = math.sqrt(height / Z)  # Non-linear scaling
        # Account for spatial isolation (harder to build isolated blocks)
        neighbors_with_blocks = sum(1 for dx, dy in [(-1,0), (1,0), (0,-1), (0,1)]
                                  if (x+dx, y+dy) in building_blocks)
        isolation_penalty = 1.0 - 0.2 * neighbors_with_blocks
        
        sequence_weight = min(0.4 + 0.6 * complexity * isolation_penalty, 1.0)
        G.add_node(f'build_seq_{x}_{y}', type=1, weight=sequence_weight)
    
    # 4. Movement collision constraints - for high-traffic areas
    # Focus on positions that are likely bottlenecks
    for x in range(X):
        for y in range(Y):
            # Check if this position is a potential bottleneck
            neighbor_targets = sum(1 for dx, dy in [(-1,0), (1,0), (0,-1), (0,1)]
                                 if 0 <= x+dx < X and 0 <= y+dy < Y and (x+dx, y+dy) in building_blocks)
            
            if neighbor_targets >= 2:  # Bottleneck position
                traffic_density = neighbor_targets / 4.0
                G.add_node(f'collision_{x}_{y}', type=1, weight=traffic_density)
    
    # Edges: Relationships between positions and constraints
    
    # 1. Position-to-agent-capacity edges
    for x in range(X):
        for y in range(Y):
            pos_node = f'pos_{x}_{y}'
            # Distance-based weight (closer positions compete more for agents)
            center_x, center_y = X//2, Y//2
            distance = math.sqrt((x - center_x)**2 + (y - center_y)**2)
            max_distance = math.sqrt(center_x**2 + center_y**2)
            competition_weight = 1.0 - distance / max(max_distance, 1)
            G.add_edge(pos_node, 'agent_capacity', weight=competition_weight)
    
    # 2. Position-to-time edges for building targets
    for (x, y), height in building_blocks.items():
        pos_node = f'pos_{x}_{y}'
        # Connect to relevant time steps based on construction sequence
        for t in range(min(T, height + 2)):  # Need time for ramp building
            time_dependency = math.exp(-0.5 * t)  # Earlier steps more critical
            G.add_edge(pos_node, f'time_{t}', weight=time_dependency)
    
    # 3. Position-to-building-sequence edges
    for (x, y), height in building_blocks.items():
        pos_node = f'pos_{x}_{y}'
        seq_node = f'build_seq_{x}_{y}'
        G.add_edge(pos_node, seq_node, weight=1.0)
        
        # Connect to neighboring positions (for ramp construction)
        for dx, dy in [(-1,0), (1,0), (0,-1), (0,1)]:
            nx_pos, ny_pos = x + dx, y + dy
            if 0 <= nx_pos < X and 0 <= ny_pos < Y:
                neighbor_node = f'pos_{nx_pos}_{ny_pos}'
                # Weight by height difference impact
                neighbor_height = building_blocks.get((nx_pos, ny_pos), 0)
                height_diff_impact = abs(height - neighbor_height) / Z
                ramp_weight = 0.3 + 0.7 * height_diff_impact
                G.add_edge(neighbor_node, seq_node, weight=ramp_weight)
    
    # 4. Position-to-collision constraint edges
    for x in range(X):
        for y in range(Y):
            collision_node = f'collision_{x}_{y}'
            if collision_node in G.nodes():
                # Connect the collision position and its neighbors
                for dx, dy in [(0,0), (-1,0), (1,0), (0,-1), (0,1)]:
                    nx_pos, ny_pos = x + dx, y + dy
                    if 0 <= nx_pos < X and 0 <= ny_pos < Y:
                        pos_node = f'pos_{nx_pos}_{ny_pos}'
                        # Central position has highest collision risk
                        collision_risk = 1.0 if (dx, dy) == (0, 0) else 0.6
                        G.add_edge(pos_node, collision_node, weight=collision_risk)
    
    # 5. Add spatial adjacency edges between neighboring positions
    for x in range(X):
        for y in range(Y):
            pos_node = f'pos_{x}_{y}'
            for dx, dy in [(-1,0), (1,0), (0,-1), (0,1)]:
                nx_pos, ny_pos = x + dx, y + dy
                if 0 <= nx_pos < X and 0 <= ny_pos < Y:
                    neighbor_node = f'pos_{nx_pos}_{ny_pos}'
                    
                    # Weight by construction coordination needs
                    pos_height = building_blocks.get((x, y), 0)
                    neighbor_height = building_blocks.get((nx_pos, ny_pos), 0)
                    
                    if pos_height > 0 or neighbor_height > 0:
                        # Higher weight for construction coordination
                        coordination_need = (pos_height + neighbor_height) / (2 * Z)
                        G.add_edge(pos_node, neighbor_node, weight=0.4 + 0.6 * coordination_need)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()