#!/usr/bin/env python3
"""
Graph converter for MagicSeq problem.
Converter created with subagent_prompt.md v_02

This problem is about finding magic sequences where each X[i] equals the count of occurrences of value i in the sequence.
Key challenges: Self-referential constraints where the value determines its own count, creating complex interdependencies.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Magic Sequence problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with positions and count constraints
    - Positions are variables that need values (type 0)
    - Count constraints ensure X[i] = count of i in sequence (type 1)  
    - Self-reference creates complex interdependencies
    - Weight positions by centrality and constraints by tightness
    """
    n = json_data.get('n', 0)
    
    if n == 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Variable nodes: sequence positions X[0], X[1], ..., X[n-1]
    for i in range(n):
        # Position weight based on centrality and potential impact
        # Central positions have more flexibility, edge positions are more constrained
        centrality = 1.0 - abs(i - n//2) / max(n//2, 1)
        
        # Also consider the range of possible values at this position
        # In magic sequences, most positions will have value 0, few have higher values
        # Use logarithmic scaling to emphasize the difficulty of non-zero assignments
        expected_sparsity = math.exp(-i/max(n/5, 1))  # Exponential decay
        weight = 0.3 * centrality + 0.7 * expected_sparsity
        
        G.add_node(f'pos_{i}', type=0, weight=min(weight, 1.0))
    
    # Constraint nodes: count constraints for each value 0..n-1
    for value in range(n):
        # Constraint tightness depends on the expected frequency of this value
        # Value 0 appears most frequently, higher values less so
        if value == 0:
            # Most positions will be 0, so this constraint is less tight
            tightness = 0.3
        elif value == 1:
            # Typically 1-2 positions have value 1
            tightness = 0.6
        elif value == 2:
            # Very few positions can have value 2
            tightness = 0.8
        else:
            # Higher values are very constrained
            tightness = 0.9 + 0.1 * min(value / n, 1.0)
        
        G.add_node(f'count_{value}', type=1, weight=min(tightness, 1.0))
    
    # Bipartite edges: position participates in count constraint
    for pos in range(n):
        for value in range(n):
            # Edge weight represents the strength of the relationship
            # Self-referential positions (pos == value) create strongest constraints
            if pos == value:
                # Self-reference: X[i] directly determines count of value i
                weight = 1.0
            else:
                # Indirect relationship: position can contribute to count
                # Weight decreases with "distance" between position and value
                distance = abs(pos - value)
                weight = math.exp(-2.0 * distance / n)  # Exponential decay
            
            # Only add edge if weight is significant
            if weight > 0.1:
                G.add_edge(f'pos_{pos}', f'count_{value}', weight=weight)
    
    # Add conflict edges between positions that compete for the same values
    # In magic sequences, there are implicit constraints about total counts
    total_sum = n  # Sum of all X[i] must equal n (total positions)
    
    # Create a global sum constraint
    G.add_node('sum_constraint', type=1, weight=0.9)  # High tightness
    
    # Connect all positions to sum constraint
    for pos in range(n):
        # Weight by potential contribution to sum
        max_contribution = min(n, n-1)  # Max value a position can have
        expected_contribution = 1.0  # Most positions contribute 0, but some contribute more
        weight = expected_contribution / max_contribution if max_contribution > 0 else 0.5
        G.add_edge(f'pos_{pos}', 'sum_constraint', weight=weight)
    
    # Add interdependency edges between positions with strong mutual constraints
    # These represent the fact that assigning a value to one position affects others
    for i in range(n):
        for j in range(i+1, min(i+3, n)):  # Only nearby positions to avoid too many edges
            if i != j:
                # Mutual dependency strength based on how values can interfere
                dependency = math.exp(-abs(i-j)/2.0) * 0.7
                if dependency > 0.3:
                    G.add_edge(f'pos_{i}', f'pos_{j}', weight=dependency)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()