#!/usr/bin/env python3
"""
Graph converter for mapping problem.
Converter created with subagent_prompt.md v_02

This problem is about mapping H.263 encoder actors onto processors in a mesh network-on-chip (NoC).
Key challenges: Communication bandwidth constraints, processor load balancing, minimizing network congestion.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the NoC mapping problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model processors, actors, flows, and network links as nodes.
    - Processors (type 2): Resource nodes with capacity constraints
    - Actors (type 0): Variable nodes that need placement
    - Flows (type 1): Constraint nodes representing communication requirements
    - Network links (type 1): Constraint nodes representing bandwidth limits
    - Communication patterns determine edge weights
    """
    # Extract problem dimensions
    row = json_data.get('row', 2)
    col = json_data.get('col', 2)
    k = row * col  # number of processors
    no_actors = json_data.get('no_actors', 0)
    no_flows = json_data.get('no_flows', 0)
    link_bandwidth = json_data.get('link_bandwidth', 1)
    processor_load = json_data.get('processor_load', 1)
    
    # Extract arrays
    actor_load = json_data.get('actor_load', [])
    inStream = json_data.get('inStream', [])
    source_destination_actor = json_data.get('source_destination_actor', [])
    arc = json_data.get('arc', [])
    unit_cost = json_data.get('unit_cost', [])
    
    G = nx.Graph()
    
    # Add processor nodes (type 2 - resource nodes)
    for p in range(k):
        # Weight by scarcity - processors in corner/edge positions are less connected
        mesh_row = p // col
        mesh_col = p % col
        
        # Calculate connectivity (number of neighbors in mesh)
        neighbors = 0
        if mesh_row > 0: neighbors += 1  # up
        if mesh_row < row - 1: neighbors += 1  # down
        if mesh_col > 0: neighbors += 1  # left
        if mesh_col < col - 1: neighbors += 1  # right
        
        # Less connected processors are more constrained (higher weight)
        connectivity_weight = 1.0 - (neighbors / 4.0)
        
        # Also consider load capacity
        capacity_weight = 1.0 - math.exp(-processor_load / 100.0)
        
        final_weight = (connectivity_weight + capacity_weight) / 2.0
        G.add_node(f'processor_{p}', type=2, weight=final_weight)
    
    # Add actor nodes (type 0 - variable nodes that need placement)
    max_actor_load = max(actor_load) if actor_load else 1
    for a in range(no_actors):
        load = actor_load[a] if a < len(actor_load) else 1
        # Weight by computational demand
        weight = load / max_actor_load if max_actor_load > 0 else 0.5
        G.add_node(f'actor_{a}', type=0, weight=weight)
    
    # Add flow constraint nodes (type 1 - communication constraints)
    max_stream = max(inStream) if inStream else 1
    for f in range(no_flows):
        stream_size = inStream[f] if f < len(inStream) else 1
        # Weight by communication intensity - larger flows are more constraining
        intensity = stream_size / max_stream if max_stream > 0 else 0.5
        
        # Apply non-linear scaling for high-bandwidth flows
        weight = 1.0 - math.exp(-3.0 * intensity)
        G.add_node(f'flow_{f}', type=1, weight=weight)
    
    # Add network link constraint nodes (type 1 - bandwidth constraints)
    # Extract unique links from arc array
    links = set()
    if arc:
        for i in range(0, len(arc), 2):
            if i + 1 < len(arc):
                src, dst = arc[i], arc[i + 1]
                if src <= k and dst <= k and src != dst:  # Only processor-to-processor links
                    links.add((min(src - 1, dst - 1), max(src - 1, dst - 1)))  # Convert to 0-based
    
    link_id = 0
    for src, dst in links:
        # Weight by potential congestion - links between central processors are more critical
        src_centrality = 1.0 - (abs(src // col - row // 2) + abs(src % col - col // 2)) / max(row, col)
        dst_centrality = 1.0 - (abs(dst // col - row // 2) + abs(dst % col - col // 2)) / max(row, col)
        centrality = (src_centrality + dst_centrality) / 2.0
        
        # Higher centrality means higher constraint weight
        weight = 0.3 + 0.7 * centrality
        G.add_node(f'link_{link_id}', type=1, weight=weight)
        
        # Connect link to the processors it connects
        G.add_edge(f'link_{link_id}', f'processor_{src}', weight=0.8)
        G.add_edge(f'link_{link_id}', f'processor_{dst}', weight=0.8)
        link_id += 1
    
    # Connect actors to flows they participate in
    # Parse source_destination_actor as pairs
    flow_connections = []
    for i in range(0, len(source_destination_actor), 2):
        if i + 1 < len(source_destination_actor):
            src_actor = source_destination_actor[i] - 1  # Convert to 0-based
            dst_actor = source_destination_actor[i + 1] - 1
            flow_connections.append((src_actor, dst_actor))
    
    for f in range(min(no_flows, len(flow_connections))):
        src_actor, dst_actor = flow_connections[f]
        
        if 0 <= src_actor < no_actors and 0 <= dst_actor < no_actors:
            # Connect source actor to flow
            stream_size = inStream[f] if f < len(inStream) else 1
            normalized_stream = min(stream_size / link_bandwidth, 1.0) if link_bandwidth > 0 else 0.5
            edge_weight = 0.4 + 0.6 * normalized_stream
            
            G.add_edge(f'actor_{src_actor}', f'flow_{f}', weight=edge_weight)
            G.add_edge(f'actor_{dst_actor}', f'flow_{f}', weight=edge_weight)
    
    # Connect actors to processors (potential mapping edges)
    # Weight edges based on actor load vs processor capacity
    for a in range(no_actors):
        actor_load_val = actor_load[a] if a < len(actor_load) else 1
        for p in range(k):
            # Weight by how well the actor fits on the processor
            load_ratio = actor_load_val / processor_load if processor_load > 0 else 0.5
            
            # Use exponential decay for high load ratios
            fit_weight = math.exp(-2.0 * load_ratio)
            
            # Only add edges if there's meaningful capacity
            if fit_weight > 0.1:
                G.add_edge(f'actor_{a}', f'processor_{p}', weight=fit_weight)
    
    # Connect flows to network links (potential routing)
    for f in range(no_flows):
        stream_size = inStream[f] if f < len(inStream) else 1
        for link_idx in range(link_id):
            # Weight by bandwidth utilization
            utilization = min(stream_size / link_bandwidth, 1.0) if link_bandwidth > 0 else 0.5
            
            # Non-linear scaling for bandwidth pressure
            pressure_weight = 1.0 - math.exp(-2.0 * utilization)
            
            if pressure_weight > 0.2:
                G.add_edge(f'flow_{f}', f'link_{link_idx}', weight=pressure_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()