#!/usr/bin/env python3
"""
Graph converter for mapping_1 problem.
Converter created with subagent_prompt.md v_02

This problem is about mapping H.263 encoder streaming applications onto 
a multiprocessor Network-on-Chip (NoC) system. The key challenge is to 
minimize the maximum processor load while considering communication costs 
between processors through the network topology.

Key challenges: 
- Balancing computational load across processors
- Minimizing communication costs through network links
- Satisfying bandwidth constraints on network links
- Mapping actors (computation tasks) to processors efficiently
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the NoC mapping problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the NoC mapping problem
    - Processors (type 0): Decision variables for where to map actors
    - Actors (type 0): Computational tasks that need mapping
    - Communication flows (type 1): Data flows between actors
    - Network links (type 1): Physical connections constraining communication
    - Resource constraints (type 1): Processor load and bandwidth limits
    
    Key entities:
    - Processors in row×col mesh topology
    - Actors with computational loads
    - Communication flows with bandwidth requirements
    - Network links with bandwidth constraints
    
    What makes instances hard:
    - High actor loads relative to processor capacity
    - High communication bandwidth relative to link capacity
    - Dense communication patterns
    - Unbalanced actor loads
    """
    
    # Extract problem parameters
    row = json_data.get('row', 1)
    col = json_data.get('col', 1)
    k = row * col  # number of processors
    no_actors = json_data.get('no_actors', 0)
    no_flows = json_data.get('no_flows', 0)
    link_bandwidth = json_data.get('link_bandwidth', 1)
    processor_load = json_data.get('processor_load', 1)
    
    # Get arrays
    inStream = json_data.get('inStream', [])
    actor_load = json_data.get('actor_load', [])
    source_destination_actor = json_data.get('source_destination_actor', [])
    unit_cost = json_data.get('unit_cost', [])
    arc = json_data.get('arc', [])
    
    G = nx.Graph()
    
    # Add processor nodes (type 0) with position-based weights
    for i in range(k):
        row_pos = i // col
        col_pos = i % col
        # Central processors are more valuable (higher connectivity)
        centrality = 1.0 - (abs(row_pos - row//2) + abs(col_pos - col//2)) / max(row + col, 1)
        # Weight by relative capacity vs average actor load
        avg_actor_load = sum(actor_load) / max(len(actor_load), 1) if actor_load else 0
        capacity_ratio = processor_load / max(avg_actor_load, 1)
        # Combine centrality and capacity with non-linear scaling
        weight = 0.3 + 0.7 * math.sqrt(centrality * min(capacity_ratio / 5.0, 1.0))
        G.add_node(f'proc_{i}', type=0, weight=weight)
    
    # Add actor nodes (type 0) with load-based weights
    max_actor_load = max(actor_load) if actor_load else 1
    for i in range(no_actors):
        load = actor_load[i] if i < len(actor_load) else 0
        # High-load actors are more critical for mapping decisions
        load_ratio = load / max_actor_load
        # Use logarithmic scaling for load importance
        weight = 0.2 + 0.8 * math.log(1 + 9 * load_ratio) / math.log(10)
        G.add_node(f'actor_{i}', type=0, weight=weight)
    
    # Add communication flow constraint nodes (type 1)
    max_stream = max(inStream) if inStream else 1
    for i in range(no_flows):
        stream_size = inStream[i] if i < len(inStream) else 0
        # Larger flows create more constraint pressure
        stream_ratio = stream_size / max_stream
        # Communication constraints are critical - use exponential scaling
        weight = 0.4 + 0.6 * (1.0 - math.exp(-3.0 * stream_ratio))
        G.add_node(f'flow_{i}', type=1, weight=weight)
    
    # Add network link constraint nodes (type 1) based on arc structure
    # Parse arc array (assuming it's [from1, to1, from2, to2, ...])
    no_links = (len(arc) // 2) if arc else 0
    link_weights = []
    for i in range(0, len(arc), 2):
        if i + 1 < len(arc):
            from_proc = arc[i] - 1  # Convert to 0-based
            to_proc = arc[i + 1] - 1
            if 0 <= from_proc < k and 0 <= to_proc < k:
                # Calculate total communication demand on this link
                total_demand = sum(inStream[j] for j in range(no_flows) 
                                 if j * 2 < len(source_destination_actor) and 
                                    j * 2 + 1 < len(source_destination_actor))
                # Link utilization determines constraint tightness
                utilization = total_demand / max(link_bandwidth * no_flows, 1)
                # High utilization makes links more constraining
                weight = 0.3 + 0.7 * min(utilization, 1.0)
                link_weights.append(weight)
                G.add_node(f'link_{i//2}', type=1, weight=weight)
    
    # Add processor capacity constraint nodes (type 1)
    total_actor_load = sum(actor_load)
    avg_load_per_proc = total_actor_load / max(k, 1)
    capacity_pressure = avg_load_per_proc / max(processor_load, 1)
    # Capacity constraints become critical when load approaches capacity
    capacity_weight = 0.5 + 0.5 * min(capacity_pressure, 1.0)
    for i in range(k):
        G.add_node(f'proc_capacity_{i}', type=1, weight=capacity_weight)
    
    # Add edges: actors to flow constraints they participate in
    for i in range(no_flows):
        if i * 2 + 1 < len(source_destination_actor):
            src_actor = source_destination_actor[i * 2] - 1  # Convert to 0-based
            dst_actor = source_destination_actor[i * 2 + 1] - 1
            
            if 0 <= src_actor < no_actors:
                # Source actor participates in this flow
                stream_size = inStream[i] if i < len(inStream) else 0
                participation_weight = min(stream_size / max_stream, 1.0)
                G.add_edge(f'actor_{src_actor}', f'flow_{i}', weight=participation_weight)
            
            if 0 <= dst_actor < no_actors:
                # Destination actor participates in this flow
                stream_size = inStream[i] if i < len(inStream) else 0
                participation_weight = min(stream_size / max_stream, 1.0)
                G.add_edge(f'actor_{dst_actor}', f'flow_{i}', weight=participation_weight)
    
    # Add edges: processors to their capacity constraints
    for i in range(k):
        G.add_edge(f'proc_{i}', f'proc_capacity_{i}', weight=1.0)
    
    # Add edges: actors to processor capacity constraints (potential mappings)
    for i in range(no_actors):
        actor_load_val = actor_load[i] if i < len(actor_load) else 0
        load_ratio = actor_load_val / max(processor_load, 1)
        # Higher load actors have stronger impact on capacity
        impact_weight = min(load_ratio * 2.0, 1.0)
        
        for j in range(k):
            G.add_edge(f'actor_{i}', f'proc_capacity_{j}', weight=impact_weight)
    
    # Add edges: flows to network links they might use
    for i in range(no_flows):
        flow_demand = inStream[i] if i < len(inStream) else 0
        demand_ratio = flow_demand / max(link_bandwidth, 1)
        
        for j in range(len(link_weights)):
            # Flow can potentially use any link
            usage_weight = min(demand_ratio, 1.0)
            if f'link_{j}' in G:
                G.add_edge(f'flow_{i}', f'link_{j}', weight=usage_weight)
    
    # Add conflict edges between high-demand actors competing for processors
    for i in range(no_actors):
        for j in range(i + 1, no_actors):
            load_i = actor_load[i] if i < len(actor_load) else 0
            load_j = actor_load[j] if j < len(actor_load) else 0
            combined_load = load_i + load_j
            
            # Add conflict if combined load exceeds processor capacity
            if combined_load > processor_load:
                conflict_intensity = (combined_load - processor_load) / max(processor_load, 1)
                conflict_weight = min(conflict_intensity, 1.0)
                G.add_edge(f'actor_{i}', f'actor_{j}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()