#!/usr/bin/env python3
"""
Graph converter for Mario routing problem.
Created using subagent_prompt.md version: v_02

This problem is about finding the optimal path for Mario to collect gold from houses
while respecting fuel constraints. It's a constrained path optimization problem.
Key challenges: balancing fuel consumption vs gold collection, finding valid paths from Mario to Luigi.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Mario routing problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as a bipartite graph with house nodes and routing constraints
    - House nodes (type 0): Each house with gold/fuel efficiency weight
    - Fuel constraint node (type 1): Global fuel limitation
    - Path constraint nodes (type 1): Routing requirements (start/end points)
    - Edge weights represent fuel consumption and routing feasibility
    """
    # Extract problem data
    nb_houses = json_data.get('nbHouses', 0)
    mario_house = json_data.get('MarioHouse', 1)
    luigi_house = json_data.get('LuigiHouse', 2)
    fuel_max = json_data.get('fuelMax', 1)
    gold_total = json_data.get('goldTotalAmount', 1)
    gold_in_house = json_data.get('goldInHouse', [])
    conso = json_data.get('conso', [])
    
    # Reshape consumption matrix (it's stored as flat array)
    conso_matrix = []
    for i in range(nb_houses):
        row = []
        for j in range(nb_houses):
            idx = i * nb_houses + j
            if idx < len(conso):
                row.append(conso[idx])
            else:
                row.append(0)
        conso_matrix.append(row)
    
    G = nx.Graph()
    
    # House nodes (type 0) - weight by gold efficiency
    max_gold = max(gold_in_house) if gold_in_house else 1
    avg_fuel_cost = sum(sum(row) for row in conso_matrix) / (nb_houses * nb_houses) if nb_houses > 0 else 1
    
    for i in range(nb_houses):
        house_idx = i + 1  # Houses are 1-indexed in problem
        gold = gold_in_house[i] if i < len(gold_in_house) else 0
        
        # Calculate average fuel cost from this house
        avg_out_cost = sum(conso_matrix[i]) / nb_houses if nb_houses > 0 else 1
        
        # Weight by gold efficiency (gold per unit fuel cost)
        if avg_out_cost > 0:
            efficiency = gold / avg_out_cost
            # Normalize using log scaling for better distribution
            weight = min(1.0, math.log(1 + efficiency) / math.log(1 + max_gold / avg_fuel_cost))
        else:
            weight = gold / max_gold if max_gold > 0 else 0.5
            
        # Special weighting for start/end houses
        if house_idx == mario_house:
            weight = max(weight, 0.8)  # Start is important
        elif house_idx == luigi_house:
            weight = max(weight, 0.9)  # End is critical
            
        G.add_node(f'house_{house_idx}', type=0, weight=weight)
    
    # Constraint nodes (type 1)
    
    # Global fuel constraint - weight by tightness
    total_min_fuel = min(sum(row) for row in conso_matrix) if conso_matrix else 0
    fuel_tightness = min(1.0, total_min_fuel / fuel_max) if fuel_max > 0 else 0.5
    G.add_node('fuel_constraint', type=1, weight=fuel_tightness)
    
    # Path constraint (must visit Mario and Luigi)
    G.add_node('path_constraint', type=1, weight=1.0)
    
    # Route efficiency constraint - measures path quality
    G.add_node('route_efficiency', type=1, weight=0.8)
    
    # Add edges from houses to constraints
    
    # Fuel constraint edges - weight by fuel consumption relative to capacity
    for i in range(nb_houses):
        house_idx = i + 1
        avg_consumption = sum(conso_matrix[i]) / nb_houses if nb_houses > 0 else 0
        fuel_weight = min(1.0, avg_consumption / fuel_max) if fuel_max > 0 else 0.5
        G.add_edge(f'house_{house_idx}', 'fuel_constraint', weight=fuel_weight)
    
    # Path constraint edges - higher weight for start/end houses
    for i in range(nb_houses):
        house_idx = i + 1
        if house_idx == mario_house or house_idx == luigi_house:
            weight = 1.0
        else:
            weight = 0.3  # Other houses less critical for path constraint
        G.add_edge(f'house_{house_idx}', 'path_constraint', weight=weight)
    
    # Route efficiency edges - weight by gold/fuel ratio
    for i in range(nb_houses):
        house_idx = i + 1
        gold = gold_in_house[i] if i < len(gold_in_house) else 0
        avg_cost = sum(conso_matrix[i]) / nb_houses if nb_houses > 0 else 1
        
        if avg_cost > 0 and gold > 0:
            efficiency_ratio = gold / avg_cost
            # Use exponential scaling to emphasize high-efficiency houses
            weight = min(1.0, 1.0 - math.exp(-efficiency_ratio / 10))
        else:
            weight = 0.1
            
        G.add_edge(f'house_{house_idx}', 'route_efficiency', weight=weight)
    
    # Add selective edges between houses based on fuel consumption
    # Only connect the most relevant house pairs to avoid excessive density
    for i in range(nb_houses):
        house_i = i + 1
        
        # Find closest houses by fuel cost
        neighbors = []
        for j in range(nb_houses):
            if i != j:
                house_j = j + 1
                fuel_cost = conso_matrix[i][j] if j < len(conso_matrix[i]) else float('inf')
                if fuel_cost < fuel_max * 0.3:  # More restrictive threshold
                    neighbors.append((house_j, fuel_cost))
        
        # Sort by fuel cost and only connect to closest neighbors
        neighbors.sort(key=lambda x: x[1])
        max_connections = min(8, len(neighbors))  # Limit connections per house
        
        for house_j, fuel_cost in neighbors[:max_connections]:
            if not G.has_edge(f'house_{house_i}', f'house_{house_j}'):
                # Weight by inverse fuel cost with exponential decay
                weight = math.exp(-4.0 * fuel_cost / fuel_max)
                G.add_edge(f'house_{house_i}', f'house_{house_j}', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()