#!/usr/bin/env python3
"""
Graph converter for Mario routing problem.
# Converter created with subagent_prompt.md v_02

This problem is about finding an optimal path for Mario to collect gold coins
while minimizing fuel consumption. Mario must start at his house and end at Luigi's house.
Key challenges: balancing high-value houses with fuel efficiency, path planning constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Mario routing problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with houses (variables) and constraints
    - Houses are decision points (visit or not, routing order)
    - Fuel constraint limits total path cost
    - Path constraint enforces Mario->Luigi route
    - Edge weights reflect fuel efficiency and value density
    - Node weights reflect gold value and fuel accessibility
    """
    # Extract problem data
    nb_houses = json_data.get('nbHouses', 0)
    mario_house = json_data.get('MarioHouse', 1) - 1  # Convert to 0-indexed
    luigi_house = json_data.get('LuigiHouse', 2) - 1  # Convert to 0-indexed
    fuel_max = json_data.get('fuelMax', 1)
    gold_total = json_data.get('goldTotalAmount', 1)
    gold_in_house = json_data.get('goldInHouse', [])
    conso = json_data.get('conso', [])
    
    # Reshape consumption matrix (it's flattened in JSON)
    consumption = []
    for i in range(nb_houses):
        row = []
        for j in range(nb_houses):
            idx = i * nb_houses + j
            if idx < len(conso):
                row.append(conso[idx])
            else:
                row.append(0)
        consumption.append(row)
    
    G = nx.Graph()
    
    # Add house nodes (type 0 - decision variables)
    max_gold = max(gold_in_house) if gold_in_house else 1
    for i in range(nb_houses):
        gold_value = gold_in_house[i] if i < len(gold_in_house) else 0
        
        # Weight houses by value density and accessibility
        # High-value houses get higher weights
        value_weight = gold_value / max_gold if max_gold > 0 else 0.5
        
        # Consider accessibility - houses with very high fuel costs to reach are less valuable
        min_fuel_to_reach = min(consumption[j][i] for j in range(nb_houses)) if consumption else 1
        max_fuel_to_reach = max(consumption[j][i] for j in range(nb_houses)) if consumption else 1
        if max_fuel_to_reach > min_fuel_to_reach:
            accessibility = 1.0 - (min_fuel_to_reach / max_fuel_to_reach)
        else:
            accessibility = 0.5
        
        # Combine value and accessibility with non-linear weighting
        # Use exponential to emphasize high-value, accessible houses
        combined_weight = (0.7 * value_weight + 0.3 * accessibility)
        if combined_weight > 0:
            final_weight = 1.0 - math.exp(-3.0 * combined_weight)
        else:
            final_weight = 0.1
        
        # Special handling for start/end houses
        if i == mario_house or i == luigi_house:
            final_weight = max(final_weight, 0.8)  # Always important
        
        G.add_node(f'house_{i}', type=0, weight=min(final_weight, 1.0))
    
    # Add constraint nodes (type 1)
    
    # 1. Fuel constraint - single global constraint
    # Weight by how tight the fuel constraint is
    total_min_path_fuel = min(consumption[mario_house][luigi_house], 
                             sum(min(consumption[i]) for i in range(nb_houses)) / nb_houses)
    fuel_tightness = total_min_path_fuel / fuel_max if fuel_max > 0 else 0.5
    fuel_weight = min(fuel_tightness * 2, 1.0)  # Scale up tightness
    G.add_node('fuel_constraint', type=1, weight=fuel_weight)
    
    # 2. Path constraints - one for each house position in potential path
    # These represent the ordering/connectivity constraints
    for i in range(nb_houses):
        # Weight by how constrained this position is in terms of connectivity
        outgoing_options = sum(1 for j in range(nb_houses) if consumption[i][j] <= fuel_max and i != j)
        incoming_options = sum(1 for j in range(nb_houses) if consumption[j][i] <= fuel_max and i != j)
        connectivity = (outgoing_options + incoming_options) / (2 * nb_houses)
        constraint_weight = 1.0 - connectivity  # Less connected = more constrained
        G.add_node(f'path_constraint_{i}', type=1, weight=constraint_weight)
    
    # 3. Start/End constraints
    G.add_node('start_constraint', type=1, weight=0.9)  # Must start at Mario's
    G.add_node('end_constraint', type=1, weight=0.9)    # Must end at Luigi's
    
    # Add edges: house participation in constraints
    
    # Connect all houses to fuel constraint (all consume fuel)
    for i in range(nb_houses):
        # Weight by fuel efficiency - how much fuel this house typically requires
        avg_fuel_cost = sum(consumption[i][j] + consumption[j][i] for j in range(nb_houses)) / (2 * nb_houses)
        fuel_participation = min(avg_fuel_cost / fuel_max, 1.0) if fuel_max > 0 else 0.5
        G.add_edge(f'house_{i}', 'fuel_constraint', weight=fuel_participation)
    
    # Connect houses to their path constraints
    for i in range(nb_houses):
        G.add_edge(f'house_{i}', f'path_constraint_{i}', weight=1.0)
    
    # Connect start/end houses to their special constraints
    G.add_edge(f'house_{mario_house}', 'start_constraint', weight=1.0)
    G.add_edge(f'house_{luigi_house}', 'end_constraint', weight=1.0)
    
    # Add routing edges between houses (type 0 to type 0)
    # Only add edges for feasible routes (within fuel budget)
    for i in range(nb_houses):
        for j in range(nb_houses):
            if i != j and consumption[i][j] <= fuel_max:
                # Weight by fuel efficiency and value gain
                fuel_cost = consumption[i][j]
                value_gain = gold_in_house[j] if j < len(gold_in_house) else 0
                
                # Normalize fuel cost (lower is better)
                fuel_efficiency = 1.0 - (fuel_cost / fuel_max) if fuel_max > 0 else 0.5
                
                # Normalize value gain (higher is better)
                value_ratio = value_gain / max_gold if max_gold > 0 else 0
                
                # Combine with emphasis on efficiency for route planning
                route_weight = 0.6 * fuel_efficiency + 0.4 * value_ratio
                
                # Use exponential scaling to emphasize good routes
                if route_weight > 0:
                    final_route_weight = 1.0 - math.exp(-2.0 * route_weight)
                else:
                    final_route_weight = 0.1
                
                G.add_edge(f'house_{i}', f'house_{j}', weight=min(final_route_weight, 1.0))
    
    # Add conflict edges for high-fuel routes that compete for budget
    # Find routes that would consume significant fuel
    high_fuel_threshold = fuel_max * 0.3  # Routes using >30% of fuel budget
    high_fuel_routes = []
    
    for i in range(nb_houses):
        for j in range(nb_houses):
            if i != j and consumption[i][j] > high_fuel_threshold:
                high_fuel_routes.append((i, j, consumption[i][j]))
    
    # Add conflicts between expensive routes
    for idx1, (i1, j1, cost1) in enumerate(high_fuel_routes):
        for idx2, (i2, j2, cost2) in enumerate(high_fuel_routes[idx1+1:], idx1+1):
            # Routes conflict if using both would exceed fuel budget
            if cost1 + cost2 > fuel_max * 0.8:  # Conflicts if >80% of budget
                conflict_strength = min((cost1 + cost2) / fuel_max, 1.0)
                G.add_edge(f'house_{i1}', f'house_{i2}', weight=conflict_strength)
                G.add_edge(f'house_{j1}', f'house_{j2}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()