#!/usr/bin/env python3
"""
Graph converter for Multi-dimensional Knapsack problem.
Converter created with subagent_prompt.md v_02

This problem is about selecting items to maximize value while respecting multiple capacity constraints.
Key challenges: multiple overlapping constraints create complex interactions between items,
constraint tightness varies significantly, and high-value items may have high resource consumption.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the multi-dimensional knapsack instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with items and constraints
    - Items (type 0): Weight by value density and constraint involvement
    - Constraints (type 1): Weight by tightness (demand vs capacity)
    - Edges: Model resource consumption with non-linear weights
    - Add conflict edges between items competing for tight resources
    """
    # Access data directly from json_data dict
    N = json_data.get('N', 0)  # number of items
    M = json_data.get('M', 0)  # number of constraints
    c = json_data.get('c', [])  # item values
    b = json_data.get('b', [])  # constraint capacities
    a = json_data.get('a', [])  # coefficients (flat array: M*N values)
    
    if N == 0 or M == 0:
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    # Reshape flat array 'a' into M x N matrix
    # a[i][j] = coefficient of item j in constraint i
    a_matrix = []
    for i in range(M):
        row = []
        for j in range(N):
            idx = i * N + j
            if idx < len(a):
                row.append(a[idx])
            else:
                row.append(0)
        a_matrix.append(row)
    
    G = nx.Graph()
    
    # Calculate value density and constraint involvement for items
    max_value = max(c) if c else 1
    
    # Item nodes (type 0) with value and involvement-based weights
    for j in range(N):
        value = c[j] if j < len(c) else 0
        
        # Calculate total resource consumption across all constraints
        total_consumption = sum(a_matrix[i][j] for i in range(M))
        max_consumption = max(sum(row) for row in a_matrix) if a_matrix else 1
        
        # Value density: value per unit of resource consumption
        consumption_normalized = total_consumption / max_consumption if max_consumption > 0 else 0.1
        if total_consumption > 0:
            value_density = (value / max_value) / (consumption_normalized + 0.1)
        else:
            value_density = value / max_value
        
        # Non-linear scaling: items with extreme value density get higher weights
        item_weight = 1.0 - math.exp(-2.0 * value_density)
        item_weight = max(0.1, min(0.9, item_weight))  # Clamp to [0.1, 0.9]
        
        G.add_node(f'item_{j}', type=0, weight=item_weight)
    
    # Constraint nodes (type 1) with tightness-based weights
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        
        # Calculate total demand for this constraint
        total_demand = sum(a_matrix[i][j] for j in range(N))
        
        # Constraint tightness: how much demand exceeds capacity
        if capacity > 0:
            demand_ratio = total_demand / capacity
            # Use exponential scaling for tightness
            if demand_ratio > 1.0:
                tightness = 1.0 - math.exp(-(demand_ratio - 1.0))
            else:
                tightness = 0.3 + 0.4 * demand_ratio  # [0.3, 0.7] for non-tight constraints
        else:
            tightness = 0.9  # Zero capacity is very tight
        
        tightness = max(0.1, min(0.9, tightness))
        G.add_node(f'constraint_{i}', type=1, weight=tightness)
    
    # Bipartite edges: item-constraint participation with consumption-based weights
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        for j in range(N):
            consumption = a_matrix[i][j]
            if consumption > 0:
                # Weight by relative consumption with exponential scaling
                consumption_ratio = consumption / capacity if capacity > 0 else 0.5
                # Use sqrt to compress high ratios while maintaining distinctions
                edge_weight = math.sqrt(consumption_ratio)
                edge_weight = max(0.1, min(1.0, edge_weight))
                
                G.add_edge(f'item_{j}', f'constraint_{i}', weight=edge_weight)
    
    # Add conflict edges between items competing for tight resources
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        total_demand = sum(a_matrix[i][j] for j in range(N))
        
        # Only add conflicts for oversubscribed constraints
        if total_demand > capacity * 1.2:  # 20% oversubscription threshold
            # Find items that significantly contribute to this constraint
            items_in_constraint = []
            for j in range(N):
                if a_matrix[i][j] > 0:
                    contribution = a_matrix[i][j] / capacity if capacity > 0 else 0
                    if contribution > 0.1:  # Only consider significant contributors
                        items_in_constraint.append((j, contribution))
            
            # Sort by contribution and add conflicts between top contributors
            items_in_constraint.sort(key=lambda x: x[1], reverse=True)
            max_conflicts = min(len(items_in_constraint), 8)  # Limit conflicts to avoid dense graph
            
            for idx1 in range(max_conflicts):
                for idx2 in range(idx1 + 1, max_conflicts):
                    j1, contrib1 = items_in_constraint[idx1]
                    j2, contrib2 = items_in_constraint[idx2]
                    
                    # Conflict strength based on combined consumption vs capacity
                    combined_consumption = a_matrix[i][j1] + a_matrix[i][j2]
                    if combined_consumption > capacity:
                        conflict_strength = (combined_consumption - capacity) / capacity
                        conflict_weight = min(1.0, conflict_strength * 0.5)  # Scale down conflicts
                        
                        # Only add edge if not already exists (avoid duplicate edges)
                        if not G.has_edge(f'item_{j1}', f'item_{j2}'):
                            G.add_edge(f'item_{j1}', f'item_{j2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()