#!/usr/bin/env python3
"""
Graph converter for Multi-dimensional Knapsack problem.
Created using subagent_prompt.md version: v_02

This problem is about selecting items to maximize profit while respecting multiple capacity constraints.
Key challenges: Items compete for limited capacity across multiple dimensions/constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the multi-dimensional knapsack instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with items and constraint nodes
    - Items (type 0): Decision variables with value-density weights
    - Constraints (type 1): Capacity constraints with tightness-based weights
    - Edges: Item-constraint participation with consumption ratio weights
    - Add conflict edges between high-consumption items in tight constraints
    """
    N = json_data.get('N', 0)  # number of items
    M = json_data.get('M', 0)  # number of constraints
    c = json_data.get('c', [])  # item values
    b = json_data.get('b', [])  # constraint capacities
    a_flat = json_data.get('a', [])  # flattened M×N weight matrix
    
    # Reconstruct the a matrix from flattened format
    a = []
    for j in range(M):
        row = []
        for i in range(N):
            idx = j * N + i
            if idx < len(a_flat):
                row.append(a_flat[idx])
            else:
                row.append(0)
        a.append(row)
    
    G = nx.Graph()
    
    # Item nodes (type 0) with value-density based weights
    max_value = max(c) if c else 1
    for i in range(N):
        value = c[i] if i < len(c) else 0
        # Calculate value density across all constraints
        total_weight = sum(a[j][i] if j < len(a) and i < len(a[j]) else 0 for j in range(M))
        
        if total_weight > 0:
            # Use logarithmic scaling for value density to avoid extreme values
            value_density = value / total_weight
            max_density = max_value / max(1, min(sum(row) for row in a) / N)
            weight = min(1.0, math.log(1 + value_density) / math.log(1 + max_density))
        else:
            # Items with no weight get moderate value-based weight
            weight = value / max_value if max_value > 0 else 0.5
            
        G.add_node(f'item_{i}', type=0, weight=weight)
    
    # Constraint nodes (type 1) with tightness-based weights
    for j in range(M):
        capacity = b[j] if j < len(b) else 1
        # Calculate total demand for this constraint
        total_demand = sum(a[j][i] if j < len(a) and i < len(a[j]) else 0 for i in range(N))
        
        if total_demand > 0 and capacity > 0:
            # Tightness: how close total demand is to capacity
            if total_demand > capacity:
                # Over-constrained: high tightness
                tightness = 1.0 - (capacity / total_demand)
            else:
                # Under-constrained: moderate tightness based on utilization
                utilization = total_demand / capacity
                tightness = 0.3 + 0.4 * utilization  # Range [0.3, 0.7]
        else:
            tightness = 0.5  # Default for degenerate cases
            
        G.add_node(f'constraint_{j}', type=1, weight=min(1.0, tightness))
    
    # Bipartite edges: item-constraint participation
    for j in range(M):
        capacity = b[j] if j < len(b) else 1
        if capacity <= 0:
            continue
            
        for i in range(N):
            weight_ij = a[j][i] if j < len(a) and i < len(a[j]) else 0
            if weight_ij > 0:
                # Edge weight based on consumption ratio with exponential scaling
                consumption_ratio = weight_ij / capacity
                # Use exponential function to emphasize high-consumption relationships
                edge_weight = 1.0 - math.exp(-3.0 * consumption_ratio)
                G.add_edge(f'item_{i}', f'constraint_{j}', weight=min(1.0, edge_weight))
    
    # Add conflict edges between items in over-constrained knapsacks
    for j in range(M):
        capacity = b[j] if j < len(b) else 1
        if capacity <= 0:
            continue
            
        # Calculate total demand
        total_demand = sum(a[j][i] if j < len(a) and i < len(a[j]) else 0 for i in range(N))
        
        # Only add conflicts for significantly over-constrained knapsacks
        if total_demand > capacity * 1.3:
            # Find items that consume significant capacity in this constraint
            high_consumers = []
            for i in range(N):
                weight_ij = a[j][i] if j < len(a) and i < len(a[j]) else 0
                if weight_ij > capacity * 0.1:  # Items using >10% of capacity
                    high_consumers.append((i, weight_ij))
            
            # Sort by consumption and take top items
            high_consumers.sort(key=lambda x: x[1], reverse=True)
            top_consumers = high_consumers[:min(6, len(high_consumers))]
            
            # Add conflict edges between pairs that can't both fit
            for idx1 in range(len(top_consumers)):
                for idx2 in range(idx1 + 1, len(top_consumers)):
                    i1, weight1 = top_consumers[idx1]
                    i2, weight2 = top_consumers[idx2]
                    
                    # Check if both items together exceed capacity
                    if weight1 + weight2 > capacity:
                        # Conflict strength based on how much they exceed capacity
                        excess = (weight1 + weight2 - capacity) / capacity
                        conflict_weight = min(1.0, 0.3 + 0.7 * excess)
                        G.add_edge(f'item_{i1}', f'item_{i2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()