#!/usr/bin/env python3
"""
Graph converter for Multi-Dimensional Knapsack problem.
# Converter created with subagent_prompt.md v_02

This problem is about selecting items to maximize profit while satisfying multiple capacity constraints.
Key challenges: Complex interactions between items and multiple constraints, tight capacity limits.

Strategy: Create bipartite graph with items (type 0) and constraint nodes (type 1).
- Items weighted by value density (profit/total_weight)
- Constraints weighted by tightness (1 - capacity/total_demand)
- Edges weighted by consumption ratio
- Add conflict edges between competing high-value items in tight constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the multi-dimensional knapsack instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Bipartite model capturing item-constraint interactions
    - Items as type 0 nodes with value density weights
    - Constraints as type 1 nodes with tightness weights
    - Consumption-based edge weights
    - Conflict edges for competing items in tight constraints
    """
    # Extract problem parameters
    N = json_data.get('N', 0)  # number of items
    M = json_data.get('M', 0)  # number of constraints
    c = json_data.get('c', [])  # profits
    b = json_data.get('b', [])  # capacities
    a_flat = json_data.get('a', [])  # flattened weights matrix
    
    # Reconstruct 2D array from flattened data
    # a[i][j] = weight of item j in constraint i
    a = []
    for i in range(M):
        row = []
        for j in range(N):
            idx = i * N + j
            if idx < len(a_flat):
                row.append(a_flat[idx])
            else:
                row.append(0)
        a.append(row)
    
    G = nx.Graph()
    
    # Add item nodes (type 0) with value density weights
    max_profit = max(c) if c else 1
    for j in range(N):
        profit = c[j] if j < len(c) else 0
        
        # Calculate total weight across all constraints for this item
        total_weight = sum(a[i][j] for i in range(M)) if M > 0 else 1
        
        # Value density: profit per unit total weight
        if total_weight > 0:
            value_density = profit / total_weight
            max_density = max_profit / 1  # theoretical max density
            weight = min(value_density / max_density, 1.0) if max_density > 0 else 0.5
        else:
            weight = 0.1  # Low weight for items with no resource consumption
        
        G.add_node(f'item_{j}', type=0, weight=weight)
    
    # Add constraint nodes (type 1) with tightness weights
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        
        # Calculate total demand for this constraint
        total_demand = sum(a[i][j] for j in range(N))
        
        # Tightness: how much the total demand exceeds capacity
        if total_demand > 0 and capacity > 0:
            if total_demand > capacity:
                # Oversubscribed constraint - use exponential scaling for high tightness
                ratio = total_demand / capacity
                tightness = 1.0 - math.exp(-2.0 * (ratio - 1.0))
            else:
                # Undersubscribed - linear scaling
                tightness = 0.3 + 0.2 * (total_demand / capacity)
        else:
            tightness = 0.5
        
        G.add_node(f'constraint_{i}', type=1, weight=min(tightness, 1.0))
    
    # Add bipartite edges: item-constraint participation
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        for j in range(N):
            weight_ij = a[i][j] if i < len(a) and j < len(a[i]) else 0
            
            if weight_ij > 0:  # Only connect if item uses this constraint
                # Edge weight based on consumption ratio
                consumption_ratio = weight_ij / capacity if capacity > 0 else 0.5
                edge_weight = min(consumption_ratio * 2.0, 1.0)  # Scale to emphasize high consumption
                
                G.add_edge(f'item_{j}', f'constraint_{i}', weight=edge_weight)
    
    # Add conflict edges between high-value competing items in tight constraints
    for i in range(M):
        capacity = b[i] if i < len(b) else 1
        total_demand = sum(a[i][j] for j in range(N))
        
        # Only add conflicts for significantly oversubscribed constraints
        if total_demand > capacity * 1.3:
            # Find items that significantly contribute to this constraint
            significant_items = []
            for j in range(N):
                weight_ij = a[i][j] if i < len(a) and j < len(a[i]) else 0
                profit = c[j] if j < len(c) else 0
                
                # Consider items that use substantial capacity and have decent profit
                if weight_ij > capacity * 0.05 and profit > 0:
                    efficiency = profit / weight_ij if weight_ij > 0 else 0
                    significant_items.append((j, weight_ij, efficiency))
            
            # Sort by efficiency (profit per weight in this constraint)
            significant_items.sort(key=lambda x: x[2], reverse=True)
            
            # Add conflict edges between top competing items
            for idx1 in range(min(len(significant_items), 6)):  # Top 6 items
                for idx2 in range(idx1+1, min(len(significant_items), 6)):
                    j1, w1, eff1 = significant_items[idx1]
                    j2, w2, eff2 = significant_items[idx2]
                    
                    # Only add conflict if both items together exceed capacity
                    if w1 + w2 > capacity:
                        # Conflict strength based on how much they exceed capacity
                        excess = (w1 + w2 - capacity) / capacity
                        conflict_weight = min(excess, 1.0)
                        
                        # Use exponential scaling for strong conflicts
                        conflict_weight = 1.0 - math.exp(-3.0 * conflict_weight)
                        
                        G.add_edge(f'item_{j1}', f'item_{j2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()