#!/usr/bin/env python3
"""
Graph converter for model4_opt problem.
Created using subagent_prompt.md version: v_02

This problem is about hospital patient scheduling optimization.
Key challenges: capacity constraints, gender restrictions, surgeon availability, operating theater scheduling
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the hospital patient scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with patients, resources, and constraints
    - Patients (type 0): decision variables with priority/urgency weights
    - Resource nodes (type 2): rooms, surgeons, operating theaters 
    - Constraint nodes (type 1): capacity, gender, scheduling constraints
    - Edges model resource needs and constraint participation
    """
    
    # Extract problem parameters
    num_patients = json_data.get('num_patients', 0)
    num_rooms = json_data.get('num_rooms', 0) 
    num_surgeons = json_data.get('num_surgeons', 0)
    num_ot = json_data.get('num_ot', 0)
    horizon = json_data.get('horizon', 14)
    
    # Patient characteristics
    length_of_stay = json_data.get('length_of_stay', [])
    gender = json_data.get('gender', [])
    surgery_duration = json_data.get('surgery_duration', [])
    surgeon = json_data.get('surgeon', [])
    release_day = json_data.get('release_day', [])
    due_day = json_data.get('due_day', [])
    mandatory = json_data.get('mandatory', [True] * num_patients)
    incompatible_rooms = json_data.get('incompatible_rooms', [])
    
    # Resource capacities
    capacity = json_data.get('capacity', [])
    max_surgery = json_data.get('max_surgery', [])
    max_ot = json_data.get('max_ot', [])
    
    G = nx.Graph()
    
    # Patient nodes (type 0) - weight by urgency and complexity
    max_duration = max(surgery_duration) if surgery_duration else 1
    max_stay = max(length_of_stay) if length_of_stay else 1
    
    for p in range(num_patients):
        # Calculate urgency: tighter time window = higher weight
        window = due_day[p] - release_day[p] if p < len(due_day) and p < len(release_day) else horizon
        urgency = 1.0 - (window / horizon) if window > 0 else 1.0
        
        # Calculate complexity: longer surgery + longer stay = higher weight
        complexity = 0.5 * (surgery_duration[p] / max_duration if p < len(surgery_duration) else 0) + \
                    0.5 * (length_of_stay[p] / max_stay if p < len(length_of_stay) else 0)
        
        # Mandatory patients get priority boost
        priority_boost = 0.3 if (p < len(mandatory) and mandatory[p]) else 0.0
        
        patient_weight = min(0.4 * urgency + 0.4 * complexity + priority_boost, 1.0)
        G.add_node(f'patient_{p}', type=0, weight=patient_weight)
    
    # Room resource nodes (type 2) - weight by scarcity
    total_capacity = sum(capacity) if capacity else num_rooms
    for r in range(num_rooms):
        room_capacity = capacity[r] if r < len(capacity) else 1
        scarcity = 1.0 - (room_capacity / total_capacity) if total_capacity > 0 else 0.5
        G.add_node(f'room_{r}', type=2, weight=scarcity)
    
    # Surgeon resource nodes (type 2) - weight by workload variation
    for s in range(num_surgeons):
        # Calculate workload variation across days
        surgeon_schedule = []
        for d in range(horizon):
            idx = s * horizon + d
            workload = max_surgery[idx] if idx < len(max_surgery) else 0
            surgeon_schedule.append(workload)
        
        avg_workload = sum(surgeon_schedule) / len(surgeon_schedule) if surgeon_schedule else 0
        max_workload = max(surgeon_schedule) if surgeon_schedule else 1
        workload_weight = avg_workload / max_workload if max_workload > 0 else 0.5
        G.add_node(f'surgeon_{s}', type=2, weight=workload_weight)
    
    # Operating theater resource nodes (type 2)
    for o in range(num_ot):
        # Average OT availability
        ot_schedule = []
        for d in range(horizon):
            idx = o * horizon + d
            availability = max_ot[idx] if idx < len(max_ot) else 0
            ot_schedule.append(availability)
        
        avg_availability = sum(ot_schedule) / len(ot_schedule) if ot_schedule else 0
        max_availability = max(ot_schedule) if ot_schedule else 1
        availability_weight = avg_availability / max_availability if max_availability > 0 else 0.5
        G.add_node(f'ot_{o}', type=2, weight=availability_weight)
    
    # Gender constraint nodes (type 1) - one per room
    for r in range(num_rooms):
        # Count patients of each gender that could use this room
        male_patients = sum(1 for p in range(num_patients) if p < len(gender) and gender[p] == 1)
        female_patients = sum(1 for p in range(num_patients) if p < len(gender) and gender[p] == 2)
        
        # Tightness based on gender imbalance
        total_patients = male_patients + female_patients
        if total_patients > 0:
            imbalance = abs(male_patients - female_patients) / total_patients
            tightness = imbalance * 0.8 + 0.2  # Base tightness + imbalance effect
        else:
            tightness = 0.5
        
        G.add_node(f'gender_constraint_room_{r}', type=1, weight=min(tightness, 1.0))
    
    # Room compatibility constraint nodes (type 1)
    incompatible_count = 0
    for p in range(num_patients):
        for i in range(json_data.get('max_incompatible', 2)):
            idx = p * json_data.get('max_incompatible', 2) + i
            if idx < len(incompatible_rooms) and incompatible_rooms[idx] > 0:
                incompatible_count += 1
    
    # Single compatibility constraint with tightness based on incompatibilities
    if num_patients > 0:
        incompatibility_ratio = incompatible_count / (num_patients * num_rooms)
        compatibility_tightness = min(incompatibility_ratio * 2.0, 1.0)
    else:
        compatibility_tightness = 0.5
    
    G.add_node('room_compatibility_constraint', type=1, weight=compatibility_tightness)
    
    # Daily surgeon capacity constraints (type 1)
    for s in range(num_surgeons):
        for d in range(horizon):
            idx = s * horizon + d
            max_time = max_surgery[idx] if idx < len(max_surgery) else 0
            
            if max_time > 0:
                # Calculate demand for this surgeon on this day
                total_demand = sum(surgery_duration[p] for p in range(num_patients) 
                                 if p < len(surgeon) and surgeon[p] == s + 1)
                
                # Tightness based on demand vs capacity
                if total_demand > 0:
                    tightness = min(total_demand / max_time, 1.0) if max_time > 0 else 1.0
                else:
                    tightness = 0.1
                
                G.add_node(f'surgeon_capacity_{s}_day_{d}', type=1, weight=tightness)
    
    # Daily OT capacity constraints (type 1)
    for o in range(num_ot):
        for d in range(horizon):
            idx = o * horizon + d
            max_time = max_ot[idx] if idx < len(max_ot) else 0
            
            if max_time > 0:
                # Estimate demand (all patients could potentially use any OT)
                avg_surgery_duration = sum(surgery_duration) / num_patients if num_patients > 0 else 0
                estimated_demand = avg_surgery_duration * num_patients / num_ot
                
                tightness = min(estimated_demand / max_time, 1.0) if max_time > 0 else 0.5
                G.add_node(f'ot_capacity_{o}_day_{d}', type=1, weight=tightness)
    
    # Room capacity constraint per room (type 1)
    for r in range(num_rooms):
        room_capacity = capacity[r] if r < len(capacity) else 1
        # Estimate demand based on average length of stay
        avg_stay = sum(length_of_stay) / num_patients if num_patients > 0 else 1
        estimated_occupancy = num_patients * avg_stay / (horizon * num_rooms)
        
        tightness = min(estimated_occupancy / room_capacity, 1.0) if room_capacity > 0 else 1.0
        G.add_node(f'room_capacity_{r}', type=1, weight=tightness)
    
    # Edges: Patient-Resource relationships
    for p in range(num_patients):
        # Patient needs room - connect to all rooms with compatibility weight
        for r in range(num_rooms):
            # Check incompatibility
            incompatible = False
            for i in range(json_data.get('max_incompatible', 2)):
                idx = p * json_data.get('max_incompatible', 2) + i
                if idx < len(incompatible_rooms) and incompatible_rooms[idx] == r + 1:
                    incompatible = True
                    break
            
            if not incompatible:
                # Weight by length of stay (longer stay = stronger connection)
                stay_weight = length_of_stay[p] / max_stay if p < len(length_of_stay) and max_stay > 0 else 0.5
                G.add_edge(f'patient_{p}', f'room_{r}', weight=min(stay_weight, 1.0))
        
        # Patient needs surgeon
        if p < len(surgeon):
            s = surgeon[p] - 1  # Convert to 0-based index
            if 0 <= s < num_surgeons:
                duration_weight = surgery_duration[p] / max_duration if p < len(surgery_duration) and max_duration > 0 else 0.5
                G.add_edge(f'patient_{p}', f'surgeon_{s}', weight=min(duration_weight, 1.0))
        
        # Patient needs OT - connect to all OTs with duration weight
        for o in range(num_ot):
            duration_weight = surgery_duration[p] / max_duration if p < len(surgery_duration) and max_duration > 0 else 0.5
            G.add_edge(f'patient_{p}', f'ot_{o}', weight=min(duration_weight * 0.8, 1.0))
    
    # Edges: Patient-Constraint participation
    for p in range(num_patients):
        # Connect to gender constraints for all rooms
        for r in range(num_rooms):
            G.add_edge(f'patient_{p}', f'gender_constraint_room_{r}', weight=0.7)
        
        # Connect to room compatibility constraint
        G.add_edge(f'patient_{p}', 'room_compatibility_constraint', weight=0.6)
        
        # Connect to surgeon capacity constraints
        if p < len(surgeon):
            s = surgeon[p] - 1
            if 0 <= s < num_surgeons:
                for d in range(horizon):
                    constraint_name = f'surgeon_capacity_{s}_day_{d}'
                    if constraint_name in G.nodes():
                        # Higher weight if patient could be scheduled on this day
                        if (p < len(release_day) and p < len(due_day) and 
                            release_day[p] <= d <= due_day[p]):
                            edge_weight = 0.9
                        else:
                            edge_weight = 0.3
                        G.add_edge(f'patient_{p}', constraint_name, weight=edge_weight)
        
        # Connect to OT capacity constraints
        for o in range(num_ot):
            for d in range(horizon):
                constraint_name = f'ot_capacity_{o}_day_{d}'
                if constraint_name in G.nodes():
                    if (p < len(release_day) and p < len(due_day) and 
                        release_day[p] <= d <= due_day[p]):
                        edge_weight = 0.8
                    else:
                        edge_weight = 0.2
                    G.add_edge(f'patient_{p}', constraint_name, weight=edge_weight)
        
        # Connect to room capacity constraints
        for r in range(num_rooms):
            stay_weight = length_of_stay[p] / max_stay if p < len(length_of_stay) and max_stay > 0 else 0.5
            G.add_edge(f'patient_{p}', f'room_capacity_{r}', weight=min(stay_weight * 0.7, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()