#!/usr/bin/env python3
"""
Graph converter for Team Assignment problem.
Created using subagent_prompt.md version: v_02

This problem is about assigning players to teams while satisfying board constraints,
optimizing team balance (ratings), and maximizing player satisfaction (requests).
Key challenges: balancing team strength, satisfying requests, ensuring each board has one player per team.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the team assignment problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with constraint nodes
    - Player nodes (type 0): weighted by rating significance
    - Team constraint nodes (type 1): weighted by difficulty/competition
    - Board constraint nodes (type 1): weighted by rating variance
    - Request nodes (type 1): weighted by importance (double vs single)
    - Edges: player-constraint participation with strength weights
    """
    # Extract data
    players = json_data.get('players', 0)
    teams = json_data.get('teams', 1)
    boards = json_data.get('boards', 1)
    ratings = json_data.get('Rating', [])
    board_assignments = json_data.get('Board', [])
    single_requests = json_data.get('SingleRequested', [])
    double_requests = json_data.get('DoubleRequested', [])
    min_rating = json_data.get('minRating', min(ratings) if ratings else 1000)
    max_rating = json_data.get('maxRating', max(ratings) if ratings else 2000)
    
    if not ratings or not board_assignments:
        # Empty graph for invalid data
        return nx.Graph()
    
    G = nx.Graph()
    
    # Player nodes (type 0) - weighted by rating significance
    rating_range = max_rating - min_rating if max_rating > min_rating else 1
    for i in range(players):
        if i < len(ratings):
            rating = ratings[i]
            # Higher rated players are more important (non-linear scaling)
            rating_weight = math.pow((rating - min_rating) / rating_range, 0.7)
        else:
            rating_weight = 0.5
        G.add_node(f'player_{i+1}', type=0, weight=rating_weight)
    
    # Team balance constraints (type 1) - one per team
    # Weight by expected competition for high-rated players
    for t in range(teams):
        # Teams with more high-rated candidates face more competition
        high_rated_count = sum(1 for r in ratings if r > (min_rating + max_rating) / 2)
        competition = min(high_rated_count / max(teams, 1), 1.0)
        G.add_node(f'team_{t+1}', type=1, weight=competition)
    
    # Board constraints (type 1) - one per board
    # Weight by rating variance within board players
    for b in range(boards):
        board_players = [i for i in range(len(board_assignments)) if board_assignments[i] == b+1]
        if board_players:
            board_ratings = [ratings[i] for i in board_players if i < len(ratings)]
            if len(board_ratings) > 1:
                variance = sum((r - sum(board_ratings)/len(board_ratings))**2 for r in board_ratings)
                variance /= len(board_ratings)
                # Normalize variance to [0,1]
                max_possible_variance = (rating_range / 2) ** 2
                variance_weight = min(variance / max_possible_variance, 1.0) if max_possible_variance > 0 else 0.5
            else:
                variance_weight = 0.3
        else:
            variance_weight = 0.1
        G.add_node(f'board_{b+1}', type=1, weight=variance_weight)
    
    # Request constraints (type 1) - individual constraints for requests
    # Single requests
    for i in range(0, len(single_requests), 2):
        if i+1 < len(single_requests):
            p1, p2 = single_requests[i], single_requests[i+1]
            # Weight by rating similarity (similar ratings = easier to satisfy)
            if p1 <= len(ratings) and p2 <= len(ratings):
                rating_diff = abs(ratings[p1-1] - ratings[p2-1])
                similarity = 1.0 - min(rating_diff / rating_range, 1.0)
                weight = 0.3 + 0.4 * similarity  # Single requests are less important
            else:
                weight = 0.3
            G.add_node(f'single_req_{i//2}', type=1, weight=weight)
    
    # Double requests (more important)
    for i in range(0, len(double_requests), 2):
        if i+1 < len(double_requests):
            p1, p2 = double_requests[i], double_requests[i+1]
            # Weight by rating similarity
            if p1 <= len(ratings) and p2 <= len(ratings):
                rating_diff = abs(ratings[p1-1] - ratings[p2-1])
                similarity = 1.0 - min(rating_diff / rating_range, 1.0)
                weight = 0.6 + 0.4 * similarity  # Double requests are more important
            else:
                weight = 0.6
            G.add_node(f'double_req_{i//2}', type=1, weight=weight)
    
    # Edges: Player-constraint participation
    
    # Players to team constraints (all players can join any team)
    for i in range(players):
        for t in range(teams):
            # Edge weight based on rating fit for team
            player_rating = ratings[i] if i < len(ratings) else min_rating
            # Exponential decay for rating distance from ideal
            ideal_rating = (min_rating + max_rating) / 2
            rating_distance = abs(player_rating - ideal_rating) / rating_range
            weight = math.exp(-2.0 * rating_distance)
            G.add_edge(f'player_{i+1}', f'team_{t+1}', weight=weight)
    
    # Players to board constraints (only players assigned to that board)
    for i in range(len(board_assignments)):
        board = board_assignments[i]
        if board <= boards:
            # Weight by how critical this player is for this board
            player_rating = ratings[i] if i < len(ratings) else min_rating
            rating_importance = (player_rating - min_rating) / rating_range
            weight = 0.5 + 0.5 * rating_importance
            G.add_edge(f'player_{i+1}', f'board_{board}', weight=weight)
    
    # Players to request constraints
    for i in range(0, len(single_requests), 2):
        if i+1 < len(single_requests):
            p1, p2 = single_requests[i], single_requests[i+1]
            req_node = f'single_req_{i//2}'
            # Connect both players with weight based on their ratings
            for p in [p1, p2]:
                if p <= players:
                    player_rating = ratings[p-1] if p-1 < len(ratings) else min_rating
                    rating_weight = (player_rating - min_rating) / rating_range
                    weight = 0.4 + 0.4 * rating_weight
                    G.add_edge(f'player_{p}', req_node, weight=weight)
    
    for i in range(0, len(double_requests), 2):
        if i+1 < len(double_requests):
            p1, p2 = double_requests[i], double_requests[i+1]
            req_node = f'double_req_{i//2}'
            # Connect both players with higher weights (double requests more important)
            for p in [p1, p2]:
                if p <= players:
                    player_rating = ratings[p-1] if p-1 < len(ratings) else min_rating
                    rating_weight = (player_rating - min_rating) / rating_range
                    weight = 0.6 + 0.4 * rating_weight
                    G.add_edge(f'player_{p}', req_node, weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()