#!/usr/bin/env python3
"""
Graph converter for Service Function Chain (SFC) problem.
Created using subagent_prompt.md version: v_02

This problem is about deploying Virtual Network Functions (VNFs) across domains
to create a service function chain with minimal inter-domain communication cost.

Key challenges: 
- VNF placement constraints per domain
- Path connectivity between domains
- Minimizing gateway-to-gateway costs while satisfying service requirements
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the SFC problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph capturing VNF placement decisions and domain connectivity
    - Variable nodes (type 0): VNF nodes representing placement decisions
    - Constraint nodes (type 1): Domain constraints, domain link constraints, chain ordering
    - Resource nodes (type 2): Domains representing limited deployment capacity
    
    The graph captures:
    - VNF-to-domain assignment conflicts
    - Inter-domain communication costs  
    - Service chain ordering constraints
    - Domain capacity and type restrictions
    """
    # Extract problem parameters
    n_nodes = json_data.get('n_nodes', 0)
    n_domains = json_data.get('n_domains', 0)
    num_node_links = json_data.get('num_node_links', 0)
    vnflist_size = json_data.get('vnflist_size', 0)
    start_domain = json_data.get('start_domain', 1)
    target_domain = json_data.get('target_domain', 1)
    M = json_data.get('M', 100)  # Max edge cost
    
    # Arrays
    nodes = json_data.get('nodes', [])
    node_links = json_data.get('node_links', [])
    domain_link_weights = json_data.get('domain_link_weights', [])
    domain_constraints = json_data.get('domain_constraints', [])
    vnflist = json_data.get('vnflist', [])
    proximity_to_source = json_data.get('proximity_to_source', [])
    proximity_to_destination = json_data.get('proximity_to_destination', [])
    
    G = nx.Graph()
    
    # === Variable Nodes (Type 0): VNF nodes ===
    # Each VNF node represents a placement decision
    # VNF_DOMAIN_KEY = 8 (1-based) = index 7 (0-based)
    # VNF_TYPE = 2 (1-based) = index 1 (0-based)
    for i in range(n_nodes):
        if i * 8 + 7 < len(nodes):  # Ensure we have all 8 attributes
            domain_id = nodes[i * 8 + 7]  # VNF_DOMAIN_KEY index
            vnf_type = nodes[i * 8 + 1]   # VNF_TYPE index
            
            # Weight based on VNF type importance and domain criticality
            # Gateway nodes (type 9) are more critical for inter-domain communication
            # Endpoint nodes (type 10) are critical for start/end points
            type_importance = 1.0 if vnf_type in [9, 10] else 0.7  # Gateway/Endpoint vs others
            
            # Distance from start/target domains affects placement importance
            domain_distance = min(
                abs(domain_id - start_domain),
                abs(domain_id - target_domain)
            ) if n_domains > 1 else 0
            proximity_weight = math.exp(-domain_distance / max(n_domains, 1))
            
            node_weight = (type_importance * 0.7 + proximity_weight * 0.3)
            G.add_node(f'vnf_{i}', type=0, weight=min(node_weight, 1.0))
    
    # === Resource Nodes (Type 2): Domains ===
    # Each domain represents deployment capacity and location
    # Use the actual domain range 1..n_domains
    for d in range(1, n_domains + 1):
        # Count VNFs in this domain to assess load
        vnfs_in_domain = sum(1 for i in range(n_nodes) 
                           if i * 8 + 7 < len(nodes) and nodes[i * 8 + 7] == d)
        
        # Domain criticality based on start/target proximity
        is_critical = (d == start_domain or d == target_domain)
        criticality = 1.0 if is_critical else 0.5
        
        # Load factor (higher load = more constrained)
        avg_vnfs_per_domain = n_nodes / max(n_domains, 1)
        load_factor = min(vnfs_in_domain / max(avg_vnfs_per_domain, 1), 2.0) / 2.0
        
        domain_weight = criticality * 0.6 + load_factor * 0.4
        G.add_node(f'domain_{d}', type=2, weight=min(domain_weight, 1.0))
    
    # === Constraint Nodes (Type 1) ===
    
    # 1. Domain VNF type constraints
    n_domain_constraints = json_data.get('n_domain_constraints', 0)
    for i in range(n_domain_constraints):
        if i * 4 + 3 < len(domain_constraints):
            domain_id = domain_constraints[i * 4 + 0]
            vnf_type = domain_constraints[i * 4 + 1] 
            min_count = domain_constraints[i * 4 + 2]
            max_count = domain_constraints[i * 4 + 3]
            
            # Constraint tightness based on min/max requirements
            range_tightness = min_count / max(max_count, 1) if max_count > 0 else 1.0
            
            # Count available VNFs of this type in this domain
            available_vnfs = sum(1 for j in range(n_nodes)
                               if j * 8 + 7 < len(nodes) and 
                                  nodes[j * 8 + 7] == domain_id and 
                                  nodes[j * 8 + 1] == vnf_type)
            
            # Scarcity factor
            scarcity = min_count / max(available_vnfs, 1) if available_vnfs > 0 else 1.0
            
            constraint_weight = (range_tightness * 0.5 + min(scarcity, 1.0) * 0.5)
            G.add_node(f'domain_constraint_{i}', type=1, weight=constraint_weight)
    
    # 2. Inter-domain link constraints (gateway-to-gateway connections)
    gateway_links = 0
    for i in range(0, len(node_links), 2):
        if i + 1 < len(node_links):
            node1_idx = node_links[i] - 1  # Convert to 0-based
            node2_idx = node_links[i + 1] - 1
            
            if (node1_idx * 8 + 1 < len(nodes) and node2_idx * 8 + 1 < len(nodes) and
                nodes[node1_idx * 8 + 1] == 9 and nodes[node2_idx * 8 + 1] == 9):  # Both gateways
                
                domain1 = nodes[node1_idx * 8 + 7] if node1_idx * 8 + 7 < len(nodes) else 1
                domain2 = nodes[node2_idx * 8 + 7] if node2_idx * 8 + 7 < len(nodes) else 1
                
                if domain1 != domain2:  # Inter-domain link
                    # Get link cost from domain_link_weights matrix
                    cost_idx = (domain1 - 1) * n_domains + (domain2 - 1)
                    link_cost = domain_link_weights[cost_idx] if cost_idx < len(domain_link_weights) else M
                    
                    # Normalize cost and invert (high cost = high constraint weight)
                    cost_weight = link_cost / max(M, 1)
                    
                    G.add_node(f'interdomain_link_{gateway_links}', type=1, weight=cost_weight)
                    gateway_links += 1
    
    # 3. Service chain ordering constraints
    for i in range(vnflist_size - 1):
        # Each consecutive pair in vnflist creates an ordering constraint
        order_weight = 0.8  # Chain ordering is typically critical
        G.add_node(f'chain_order_{i}', type=1, weight=order_weight)
    
    # === Edges ===
    
    # 1. VNF-to-Domain assignment edges (bipartite)
    for i in range(n_nodes):
        if i * 8 + 7 < len(nodes):
            domain_id = nodes[i * 8 + 7]
            # Connect VNF to its domain with assignment weight
            G.add_edge(f'vnf_{i}', f'domain_{domain_id}', weight=0.9)
    
    # 2. VNF-to-Domain-Constraint edges (bipartite)
    for i in range(n_domain_constraints):
        if i * 4 + 3 < len(domain_constraints):
            domain_id = domain_constraints[i * 4 + 0]
            vnf_type = domain_constraints[i * 4 + 1]
            
            # Connect VNFs of matching type in matching domain to this constraint
            for j in range(n_nodes):
                if (j * 8 + 7 < len(nodes) and 
                    nodes[j * 8 + 7] == domain_id and 
                    nodes[j * 8 + 1] == vnf_type):
                    G.add_edge(f'vnf_{j}', f'domain_constraint_{i}', weight=0.8)
    
    # 3. VNF-to-Inter-domain-Link constraint edges
    gateway_links = 0
    for i in range(0, len(node_links), 2):
        if i + 1 < len(node_links):
            node1_idx = node_links[i] - 1
            node2_idx = node_links[i + 1] - 1
            
            if (node1_idx * 8 + 1 < len(nodes) and node2_idx * 8 + 1 < len(nodes) and
                nodes[node1_idx * 8 + 1] == 9 and nodes[node2_idx * 8 + 1] == 9):
                
                domain1 = nodes[node1_idx * 8 + 7] if node1_idx * 8 + 7 < len(nodes) else 1
                domain2 = nodes[node2_idx * 8 + 7] if node2_idx * 8 + 7 < len(nodes) else 1
                
                if domain1 != domain2:
                    # Connect both gateway nodes to this link constraint
                    if node1_idx < n_nodes:
                        G.add_edge(f'vnf_{node1_idx}', f'interdomain_link_{gateway_links}', weight=0.7)
                    if node2_idx < n_nodes:
                        G.add_edge(f'vnf_{node2_idx}', f'interdomain_link_{gateway_links}', weight=0.7)
                    gateway_links += 1
    
    # 4. VNF-to-Chain-Order constraint edges
    for i in range(vnflist_size - 1):
        if i < len(vnflist) and i + 1 < len(vnflist):
            curr_type = vnflist[i]
            next_type = vnflist[i + 1]
            
            # Connect VNFs that could participate in this ordering constraint
            for j in range(n_nodes):
                if j * 8 + 1 < len(nodes):
                    vnf_type = nodes[j * 8 + 1]
                    if vnf_type == curr_type or vnf_type == next_type:
                        # Weight by proximity requirements
                        prox_weight = 0.6
                        if i < len(proximity_to_source) and proximity_to_source[i]:
                            prox_weight += 0.2
                        if i < len(proximity_to_destination) and proximity_to_destination[i]:
                            prox_weight += 0.2
                        
                        G.add_edge(f'vnf_{j}', f'chain_order_{i}', weight=min(prox_weight, 1.0))
    
    # 5. Critical path conflict edges (Type 0 to Type 0)
    # Add conflicts between VNFs that compete for scarce domain resources
    for d in range(1, n_domains + 1):
        vnfs_in_domain = []
        for i in range(n_nodes):
            if i * 8 + 7 < len(nodes) and nodes[i * 8 + 7] == d:
                vnfs_in_domain.append(i)
        
        # If domain is overloaded, add conflict edges between VNFs
        domain_capacity = 5  # Assumed typical capacity
        if len(vnfs_in_domain) > domain_capacity:
            # Add conflicts between high-priority VNFs that can't all be selected
            for i in range(min(len(vnfs_in_domain), 8)):  # Limit conflicts
                for j in range(i + 1, min(len(vnfs_in_domain), 8)):
                    vnf1 = vnfs_in_domain[i]
                    vnf2 = vnfs_in_domain[j]
                    
                    # Conflict strength based on resource competition
                    competition = len(vnfs_in_domain) / domain_capacity
                    conflict_weight = min(competition / 2.0, 1.0)
                    
                    G.add_edge(f'vnf_{vnf1}', f'vnf_{vnf2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()