#!/usr/bin/env python3
"""
Graph converter for Mondoku-GCC-Model-Balance problem.
Converter created with subagent_prompt.md v_02

This problem is about coloring a grid where each row and column must contain
exactly one contiguous group of each color, while minimizing color imbalance.
Key challenges: Managing contiguity constraints, global cardinality, and balance optimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Mondoku problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with cells as variables and constraints as explicit nodes.
    - Cells are decision variables (type 0) with centrality-based weights
    - Row contiguity constraints (type 1) with scope-based weights
    - Column contiguity constraints (type 1) with scope-based weights  
    - Global cardinality constraints (type 1) with tightness-based weights
    - Colors as resources (type 2) with scarcity-based weights
    """
    W = json_data.get('W', 4)
    H = json_data.get('H', 4) 
    C = json_data.get('C', 2)
    
    G = nx.Graph()
    
    # Cell nodes (type 0) - decision variables
    # Weight by centrality and constraint involvement
    for r in range(H):
        for c in range(W):
            # Central cells are more constrained (involved in more constraint interactions)
            centrality = 1.0 - (abs(r - H//2) + abs(c - W//2)) / (H + W)
            # Cells involved in more constraints are harder to assign
            constraint_pressure = (H + W) / max(H + W, 1)  # Normalize involvement
            weight = (centrality * 0.6 + constraint_pressure * 0.4)
            G.add_node(f'cell_{r}_{c}', type=0, weight=weight)
    
    # Color resource nodes (type 2) - shared resources
    # Weight by scarcity (fewer available positions per color = higher weight)
    total_cells = W * H
    cells_per_color = total_cells / C if C > 0 else 1
    for color in range(1, C + 1):
        # More colors = less space per color = higher scarcity
        scarcity = 1.0 - math.exp(-C / 4.0)  # Non-linear scarcity based on color count
        G.add_node(f'color_{color}', type=2, weight=scarcity)
    
    # Row contiguity constraint nodes (type 1)
    # Each row needs exactly one group of each color
    for r in range(H):
        # Longer rows are harder to satisfy contiguity for all colors
        scope_complexity = min(math.log(W * C + 1) / math.log(10), 1.0)  # Log scaling, capped at 1.0
        tightness = min(C / W, 1.0) if W > 0 else 0.5  # More colors in shorter row = tighter
        weight = min((scope_complexity * 0.5 + tightness * 0.5), 1.0)
        G.add_node(f'row_contiguity_{r}', type=1, weight=weight)
    
    # Column contiguity constraint nodes (type 1)  
    for c in range(W):
        scope_complexity = min(math.log(H * C + 1) / math.log(10), 1.0)
        tightness = min(C / H, 1.0) if H > 0 else 0.5
        weight = min((scope_complexity * 0.5 + tightness * 0.5), 1.0)
        G.add_node(f'col_contiguity_{c}', type=1, weight=weight)
    
    # Global cardinality constraint nodes (type 1)
    # These enforce exactly one group per color per row/column
    for r in range(H):
        for color in range(1, C + 1):
            # Constraint tightness depends on how many cells must form exactly one group
            group_pressure = W / C if C > 0 else 1  # Average group size
            tightness = 1.0 - math.exp(-group_pressure / 3.0)  # Non-linear tightness
            G.add_node(f'row_{r}_color_{color}_gcc', type=1, weight=tightness)
    
    for c in range(W):
        for color in range(1, C + 1):
            group_pressure = H / C if C > 0 else 1
            tightness = 1.0 - math.exp(-group_pressure / 3.0)
            G.add_node(f'col_{c}_color_{color}_gcc', type=1, weight=tightness)
    
    # Balance constraint nodes (type 1) - minimize max difference
    # These are the optimization constraints that make the problem challenging
    balance_complexity = math.sqrt(C * W * H) / 10.0  # Scales with problem size
    balance_weight = min(balance_complexity, 1.0)
    
    for r in range(H):
        G.add_node(f'balance_row_{r}', type=1, weight=balance_weight)
    for c in range(W):
        G.add_node(f'balance_col_{c}', type=1, weight=balance_weight)
    
    # Bipartite edges: cells to constraints they participate in
    for r in range(H):
        for c in range(W):
            cell = f'cell_{r}_{c}'
            
            # Connect to row and column contiguity constraints
            G.add_edge(cell, f'row_contiguity_{r}', weight=1.0)
            G.add_edge(cell, f'col_contiguity_{c}', weight=1.0)
            
            # Connect to all color-specific GCC constraints for this row/column
            for color in range(1, C + 1):
                # Edge weight reflects how much this cell contributes to constraint difficulty
                contribution = 1.0 / (W * H)  # Each cell contributes equally to color assignment
                G.add_edge(cell, f'row_{r}_color_{color}_gcc', weight=contribution * C)
                G.add_edge(cell, f'col_{c}_color_{color}_gcc', weight=contribution * C)
            
            # Connect to balance constraints
            balance_contribution = 1.0 / max(W, H)  # Normalize by grid dimension
            G.add_edge(cell, f'balance_row_{r}', weight=balance_contribution)
            G.add_edge(cell, f'balance_col_{c}', weight=balance_contribution)
            
            # Connect cells to color resources they can use
            for color in range(1, C + 1):
                # All cells can use all colors, weight by resource scarcity
                resource_demand = 1.0 / total_cells  # Each cell demands 1/total of each color
                G.add_edge(cell, f'color_{color}', weight=resource_demand * C)
    
    # Add conflict edges between constraint nodes that interact
    # Row and column constraints conflict at intersections
    for r in range(H):
        for c in range(W):
            # Row contiguity and column contiguity interact at cell (r,c)
            interaction_strength = 2.0 / (W + H)  # Stronger for smaller grids
            G.add_edge(f'row_contiguity_{r}', f'col_contiguity_{c}', weight=interaction_strength)
            
            # GCC constraints interact with balance constraints
            for color in range(1, C + 1):
                balance_gcc_interaction = 1.0 / C  # More colors = weaker individual interaction
                G.add_edge(f'row_{r}_color_{color}_gcc', f'balance_row_{r}', weight=balance_gcc_interaction)
                G.add_edge(f'col_{c}_color_{color}_gcc', f'balance_col_{c}', weight=balance_gcc_interaction)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()