#!/usr/bin/env python3
"""
Graph converter for mqueens2 problem.
# Converter created with subagent_prompt.md v_02

This problem is about minimizing the number of queens on an n×n chessboard 
such that no two queens attack each other. Unlike standard N-Queens, 
this is an optimization problem seeking the minimum number of queens.

Key challenges: 
- Non-attacking constraint creates complex interactions between positions
- Optimization aspect means solution space varies significantly
- Board symmetries and position importance vary by location
- Problem difficulty grows exponentially with board size
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the mqueens2 problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the constraint structure
    - Position nodes (type 0): Each board position as a potential queen location
    - Constraint nodes (type 1): Non-attacking constraints for rows, columns, diagonals
    - Position weights reflect strategic importance (corner vs center vs edge)
    - Constraint weights reflect scope and criticality
    - Edges connect positions to constraints they participate in
    """
    n = json_data.get('n', 8)
    
    G = nx.Graph()
    
    # Position nodes (type 0) - board positions with strategic weights
    for r in range(n):
        for c in range(n):
            # Weight positions by strategic value
            # Center positions are more constraining (higher weight)
            # Use non-linear weighting for better sensitivity
            center_distance = math.sqrt((r - n//2)**2 + (c - n//2)**2)
            max_distance = math.sqrt(2 * (n//2)**2)
            
            # Exponential decay from center, normalized to [0,1]
            if max_distance > 0:
                centrality = math.exp(-2.0 * center_distance / max_distance)
            else:
                centrality = 1.0
            
            # Corner positions get slight boost as they're often strategic
            corner_bonus = 0.0
            if (r == 0 or r == n-1) and (c == 0 or c == n-1):
                corner_bonus = 0.2
            
            weight = min(centrality + corner_bonus, 1.0)
            G.add_node(f'pos_{r}_{c}', type=0, weight=weight)
    
    # Constraint nodes (type 1) - one for each constraint type with scope-based weights
    
    # Row constraints - each row can have at most one queen
    for r in range(n):
        # Weight by how constrained this row is (all rows equally constrained)
        G.add_node(f'row_{r}', type=1, weight=1.0)
    
    # Column constraints - each column can have at most one queen  
    for c in range(n):
        G.add_node(f'col_{c}', type=1, weight=1.0)
    
    # Diagonal constraints (positive slope) - each diagonal can have at most one queen
    for d in range(2*n-1):
        # Diagonal length varies - shorter diagonals are less constraining
        if d < n:
            diagonal_length = d + 1
        else:
            diagonal_length = 2*n - 1 - d
        
        # Weight by diagonal scope (longer diagonals more important)
        weight = diagonal_length / n
        G.add_node(f'diag_pos_{d}', type=1, weight=weight)
    
    # Anti-diagonal constraints (negative slope)
    for d in range(2*n-1):
        if d < n:
            diagonal_length = d + 1
        else:
            diagonal_length = 2*n - 1 - d
            
        weight = diagonal_length / n
        G.add_node(f'diag_neg_{d}', type=1, weight=weight)
    
    # Bipartite edges: Connect positions to constraints they participate in
    for r in range(n):
        for c in range(n):
            pos_node = f'pos_{r}_{c}'
            
            # Connect to row constraint
            G.add_edge(pos_node, f'row_{r}', weight=1.0)
            
            # Connect to column constraint
            G.add_edge(pos_node, f'col_{c}', weight=1.0)
            
            # Connect to positive diagonal constraint
            diag_pos_id = r - c + (n - 1)
            G.add_edge(pos_node, f'diag_pos_{diag_pos_id}', weight=1.0)
            
            # Connect to negative diagonal constraint  
            diag_neg_id = r + c
            G.add_edge(pos_node, f'diag_neg_{diag_neg_id}', weight=1.0)
    
    # Add global complexity constraint node reflecting optimization challenge
    total_positions = n * n
    total_constraints = n + n + (2*n-1) + (2*n-1)  # rows + cols + diags
    complexity = math.log(total_positions) / math.log(total_constraints) if total_constraints > 1 else 1.0
    G.add_node('optimization_complexity', type=1, weight=min(complexity, 1.0))
    
    # Connect complexity node to strategic positions with optimization-based weights
    for r in range(n):
        for c in range(n):
            pos_node = f'pos_{r}_{c}'
            # Weight reflects how much this position contributes to optimization difficulty
            # Positions that attack more squares are more impactful
            attacks = 0
            
            # Count positions this queen would attack
            # Row attacks
            attacks += n - 1
            # Column attacks  
            attacks += n - 1
            # Diagonal attacks
            attacks += min(r, c) + min(n-1-r, n-1-c)  # negative diagonal
            attacks += min(r, n-1-c) + min(n-1-r, c)  # positive diagonal
            
            max_attacks = 4 * (n - 1)  # theoretical maximum
            optimization_weight = attacks / max_attacks if max_attacks > 0 else 0.5
            
            G.add_edge(pos_node, 'optimization_complexity', weight=optimization_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()