#!/usr/bin/env python3
"""
Graph converter for mznc2009_roster_model problem.
Created using subagent_prompt.md version: v_02

This problem is about nurse rostering where workers are assigned to 5 types of shifts 
(Rest, Morning, Day, Evening, Joker) across multiple weeks with 7 days each.
Key challenges: meeting daily requirements, ensuring adequate rest periods, 
avoiding difficult shift transitions (evening-before-morning), minimizing isolated rest days.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the roster scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with explicit constraint nodes
    - Day-shift combinations as variable nodes (type 0)
    - Individual shift requirements as constraint nodes (type 1) 
    - Special constraints for rest patterns and transitions (type 1)
    - Connect variables to constraints they participate in
    - Weight by requirement tightness and shift difficulty
    """
    weeks = json_data.get('weeks', 1)
    reqt = json_data.get('reqt', [])
    
    # reqt is flattened 5x7 matrix: reqt[shift*7 + day] = requirement
    # Shifts: 1=Rest, 2=Morning, 3=Day, 4=Evening, 5=Joker
    # Days: 0=Monday, ..., 6=Sunday
    
    if len(reqt) != 35:  # 5 shifts * 7 days
        reqt = reqt + [0] * (35 - len(reqt))  # pad if needed
    
    G = nx.Graph()
    
    # Calculate total workers from maximum daily requirement
    max_daily_total = 0
    for day in range(7):
        daily_total = sum(reqt[shift*7 + day] for shift in range(5))
        max_daily_total = max(max_daily_total, daily_total)
    
    if max_daily_total == 0:
        max_daily_total = 1  # avoid division by zero
    
    # Variable nodes: day-shift combinations
    # Weight by shift difficulty and demand pressure
    shift_difficulty = {0: 0.2, 1: 0.6, 2: 0.8, 3: 0.9, 4: 0.7}  # Rest, Morn, Day, Eve, Joker
    
    for day in range(7):
        for shift in range(5):
            node_id = f'dayshift_{day}_{shift}'
            requirement = reqt[shift*7 + day]
            
            # Weight combines shift difficulty with requirement intensity
            base_difficulty = shift_difficulty[shift]
            requirement_pressure = requirement / max_daily_total if max_daily_total > 0 else 0
            
            # Non-linear combination: high requirements make difficult shifts even harder
            weight = base_difficulty * (1.0 + requirement_pressure)
            weight = min(weight, 1.0)
            
            G.add_node(node_id, type=0, weight=weight)
    
    # Constraint nodes: Individual shift requirements for each day
    for day in range(7):
        for shift in range(5):
            requirement = reqt[shift*7 + day]
            
            if requirement > 0:  # Only create constraints for non-zero requirements
                constraint_id = f'req_{day}_{shift}'
                
                # Weight by requirement tightness relative to total capacity
                # Higher requirements create tighter constraints
                tightness = requirement / max_daily_total if max_daily_total > 0 else 0.5
                
                # Apply non-linear scaling to emphasize high-demand constraints
                weight = 1.0 - math.exp(-3.0 * tightness)
                
                G.add_node(constraint_id, type=1, weight=weight)
                
                # Connect to the corresponding day-shift variable
                var_node = f'dayshift_{day}_{shift}'
                
                # Edge weight based on requirement intensity
                edge_weight = min(tightness * 1.5, 1.0)
                G.add_edge(var_node, constraint_id, weight=edge_weight)
    
    # Global constraint nodes for complex scheduling rules
    
    # Weekly rest constraint (at least 1 rest day per 7 consecutive days)
    G.add_node('weekly_rest_constraint', type=1, weight=0.9)
    for day in range(7):
        rest_node = f'dayshift_{day}_0'  # Rest shift = 0
        G.add_edge(rest_node, 'weekly_rest_constraint', weight=0.8)
    
    # Consecutive rest limit constraint (max 3 consecutive rest days)
    G.add_node('max_rest_sequence', type=1, weight=0.7)
    for day in range(7):
        rest_node = f'dayshift_{day}_0'
        # Weight higher for weekend days where long rest sequences are more likely
        weight = 0.9 if day >= 5 else 0.6  # Saturday=5, Sunday=6
        G.add_edge(rest_node, 'max_rest_sequence', weight=weight)
    
    # Evening-before-morning soft constraint
    G.add_node('evening_morning_transition', type=1, weight=0.6)
    for day in range(7):
        evening_node = f'dayshift_{day}_3'  # Evening shift = 3
        morning_next = f'dayshift_{(day+1)%7}_1'  # Morning shift = 1, next day
        
        # Connect both shifts to transition constraint
        G.add_edge(evening_node, 'evening_morning_transition', weight=0.8)
        G.add_edge(morning_next, 'evening_morning_transition', weight=0.8)
    
    # Isolated rest day soft constraint
    G.add_node('isolated_rest_constraint', type=1, weight=0.5)
    for day in range(7):
        rest_node = f'dayshift_{day}_0'
        prev_day_rest = f'dayshift_{(day-1)%7}_0'
        next_day_rest = f'dayshift_{(day+1)%7}_0'
        
        # Connect rest days to isolation constraint
        G.add_edge(rest_node, 'isolated_rest_constraint', weight=0.7)
        G.add_edge(prev_day_rest, 'isolated_rest_constraint', weight=0.4)
        G.add_edge(next_day_rest, 'isolated_rest_constraint', weight=0.4)
    
    # Add conflict edges for competing shift assignments on same day
    # (since each day needs specific total workers distributed across shifts)
    for day in range(7):
        daily_total = sum(reqt[shift*7 + day] for shift in range(5))
        
        if daily_total > max_daily_total * 0.8:  # High-demand days
            shifts_this_day = [f'dayshift_{day}_{shift}' for shift in range(5) 
                             if reqt[shift*7 + day] > 0]
            
            # Add conflict edges between shift types competing for workers
            for i, shift1 in enumerate(shifts_this_day):
                for shift2 in shifts_this_day[i+1:]:
                    # Weight by competition intensity
                    s1_req = reqt[int(shift1.split('_')[2])*7 + day]
                    s2_req = reqt[int(shift2.split('_')[2])*7 + day]
                    
                    competition = (s1_req + s2_req) / max_daily_total if max_daily_total > 0 else 0.5
                    conflict_weight = min(competition * 0.7, 1.0)
                    
                    G.add_edge(shift1, shift2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()