#!/usr/bin/env python3
"""
Graph converter for neighbours-rect problem.
Created using subagent_prompt.md version: v_02

This problem is about assigning numbers 1-5 to residents in a grid where each resident 
with number N>1 must have neighbors containing all numbers 1,2,...,N-1.
Key challenges: Grid constraint satisfaction with neighbor dependencies, optimization objective.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the neighbours-rect problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with position nodes and constraint nodes
    - Position nodes (type 0): Each grid cell (i,j) - weighted by centrality/connectivity
    - Neighbor constraint nodes (type 1): For each position, the constraint that 
      if assigned N>1, neighbors must contain 1..N-1
    - Global constraint nodes (type 1): Border constraints limiting max values
    - Edges connect positions to constraints they participate in
    - Edge weights reflect constraint tightness and position connectivity
    """
    # Access data directly from json_data dict
    n = json_data.get('n', 1)  # rows
    m = json_data.get('m', 1)  # columns
    
    # Create graph
    G = nx.Graph()
    
    # Helper function to get neighbors of position (i,j)
    def get_neighbors(i, j):
        neighbors = []
        for di, dj in [(-1,0), (1,0), (0,-1), (0,1)]:
            ni, nj = i + di, j + dj
            if 1 <= ni <= n and 1 <= nj <= m:
                neighbors.append((ni, nj))
        return neighbors
    
    # Add position nodes (type 0) - grid cells
    for i in range(1, n+1):
        for j in range(1, m+1):
            # Weight by connectivity and centrality
            neighbors = get_neighbors(i, j)
            neighbor_count = len(neighbors)
            max_neighbors = 4  # interior cell
            
            # Centrality: distance from center of grid
            center_i, center_j = (n+1)/2, (m+1)/2
            distance_from_center = abs(i - center_i) + abs(j - center_j)
            max_distance = (n-1)/2 + (m-1)/2
            centrality = 1.0 - distance_from_center / max(max_distance, 1)
            
            # Connectivity weight (normalized neighbor count)
            connectivity = neighbor_count / max_neighbors
            
            # Combined weight emphasizing both connectivity and centrality
            weight = 0.6 * connectivity + 0.4 * centrality
            
            G.add_node(f'pos_{i}_{j}', type=0, weight=weight)
    
    # Add neighbor constraint nodes (type 1) - one per position
    # These enforce the rule: if pos has value N>1, neighbors must contain 1..N-1
    for i in range(1, n+1):
        for j in range(1, m+1):
            neighbors = get_neighbors(i, j)
            neighbor_count = len(neighbors)
            
            # Weight by constraint difficulty/tightness
            # More neighbors = easier to satisfy (lower weight)
            # Fewer neighbors = harder to satisfy (higher weight)
            if neighbor_count == 0:
                tightness = 1.0  # No neighbors - hardest constraint
            else:
                # Constraint gets tighter as we need more distinct values (1..N-1)
                # with fewer neighbors to place them in
                # Use non-linear scaling - exponential difficulty increase
                base_difficulty = 1.0 / neighbor_count
                tightness = min(1.0, math.exp(base_difficulty) - 1.0)
            
            G.add_node(f'neighbor_constraint_{i}_{j}', type=1, weight=tightness)
    
    # Add border constraint nodes (type 1) - limits on border positions
    constraint_id = 0
    
    # Corner constraints (max value 3)
    corners = [(1,1), (1,m), (n,1), (n,m)]
    for i, j in corners:
        if 1 <= i <= n and 1 <= j <= m:  # Ensure corner exists in grid
            constraint_id += 1
            # High weight - very restrictive constraint
            G.add_node(f'corner_constraint_{constraint_id}', type=1, weight=0.9)
    
    # Edge constraints (max value 4) - not corners
    edge_positions = []
    # Top and bottom edges
    for j in range(1, m+1):
        if (1, j) not in corners:
            edge_positions.append((1, j))
        if (n, j) not in corners:
            edge_positions.append((n, j))
    # Left and right edges
    for i in range(2, n):  # Avoid corners already added
        edge_positions.append((i, 1))
        edge_positions.append((i, m))
    
    for i, j in edge_positions:
        if 1 <= i <= n and 1 <= j <= m:  # Ensure position exists
            constraint_id += 1
            # Medium weight - moderately restrictive
            G.add_node(f'edge_constraint_{constraint_id}', type=1, weight=0.7)
    
    # Add bipartite edges: positions to constraints they participate in
    
    # Connect each position to its neighbor constraint
    for i in range(1, n+1):
        for j in range(1, m+1):
            pos_node = f'pos_{i}_{j}'
            constraint_node = f'neighbor_constraint_{i}_{j}'
            
            # Edge weight: strength of participation
            # Higher for positions with fewer neighbors (more critical)
            neighbors = get_neighbors(i, j)
            neighbor_count = len(neighbors)
            participation_strength = 1.0 - (neighbor_count / 4.0)  # Normalized
            
            G.add_edge(pos_node, constraint_node, weight=participation_strength)
    
    # Connect positions to neighbor constraints of their neighbors
    # (since they provide the required values 1..N-1)
    for i in range(1, n+1):
        for j in range(1, m+1):
            pos_node = f'pos_{i}_{j}'
            neighbors = get_neighbors(i, j)
            
            for ni, nj in neighbors:
                neighbor_constraint = f'neighbor_constraint_{ni}_{nj}'
                # Weight by reciprocal relationship strength
                mutual_neighbors = len(set(get_neighbors(i, j)) & set(get_neighbors(ni, nj)))
                relationship_strength = 0.3 + 0.7 * (mutual_neighbors / 4.0)
                
                G.add_edge(pos_node, neighbor_constraint, weight=relationship_strength)
    
    # Connect positions to border constraints
    constraint_id = 0
    
    # Corner constraint connections
    for i, j in corners:
        if 1 <= i <= n and 1 <= j <= m:
            constraint_id += 1
            pos_node = f'pos_{i}_{j}'
            constraint_node = f'corner_constraint_{constraint_id}'
            # High weight - strong restriction
            G.add_edge(pos_node, constraint_node, weight=0.95)
    
    # Edge constraint connections
    for i, j in edge_positions:
        if 1 <= i <= n and 1 <= j <= m:
            constraint_id += 1
            pos_node = f'pos_{i}_{j}'
            constraint_node = f'edge_constraint_{constraint_id}'
            # Medium-high weight - moderate restriction
            G.add_edge(pos_node, constraint_node, weight=0.8)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()