#!/usr/bin/env python3
"""
Graph converter for nmseq (Magic Sequence) problem.
Created using subagent_prompt.md version: v_02

This problem is about finding a magic sequence where s[i] equals the count of value (i-1) in the sequence.
Key challenges: Many reification constraints (O(n^2) complexity), interdependent counting constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the magic sequence problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the complex interdependent structure of magic sequences
    - Variables: Each position in the sequence (type 0)
    - Constraints: Each counting constraint s[i] == count(i-1, s) (type 1)
    - Edge weights reflect complexity of counting constraints
    - Central positions have higher interdependency
    """
    n = json_data.get('n', 0)
    
    if n == 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Variable nodes: sequence positions s[0] to s[n-1]
    # Weight by centrality - middle positions are more constrained due to value range restrictions
    for i in range(n):
        # Central positions are more constrained as they have tighter value ranges in practice
        # Use exponential decay from center for non-linear weighting
        center_distance = abs(i - n/2) / (n/2) if n > 1 else 0
        centrality_weight = math.exp(-2.0 * center_distance)  # Non-linear: central positions weighted higher
        G.add_node(f'pos_{i}', type=0, weight=centrality_weight)
    
    # Constraint nodes: counting constraints for each position
    # Each constraint: s[i] == count(value=i-1, sequence=s)
    for i in range(n):
        value_being_counted = i - 1  # We're counting occurrences of (i-1)
        
        # Constraint complexity based on:
        # 1. How many positions could potentially hold this value
        # 2. Position in sequence (edge positions are typically easier)
        
        # Valid domain for the value being counted
        if value_being_counted < 0:
            # Counting -1 (impossible value), this constraint is typically easier
            complexity = 0.3
        elif value_being_counted >= n:
            # Counting values >= n (impossible), constraint is trivial  
            complexity = 0.3
        else:
            # Real counting constraint - complexity increases with:
            # - Distance from expected "natural" values (small values are more common)
            # - Central positions typically have more complex interactions
            expected_frequency = 1.0 / n  # Uniform expectation
            actual_possible_frequency = min(value_being_counted + 1, n) / n
            
            # Non-linear complexity based on deviation from expected
            deviation = abs(actual_possible_frequency - expected_frequency) / expected_frequency
            complexity = 0.5 + 0.4 * math.exp(-deviation)  # Non-linear scaling
        
        G.add_node(f'count_constraint_{i}', type=1, weight=min(complexity, 1.0))
    
    # Bipartite edges: variable participation in constraints
    # Each position variable participates in ALL counting constraints
    # (because each constraint counts occurrences across the entire sequence)
    for pos_i in range(n):
        for constraint_j in range(n):
            value_being_counted = constraint_j - 1
            
            # Edge weight represents the "influence" of position pos_i on constraint constraint_j
            # Higher weight if this position is more likely to affect the constraint
            
            if value_being_counted < 0 or value_being_counted >= n:
                # Position cannot hold this value, but still participates in counting
                participation_weight = 0.1
            else:
                # Position could hold this value
                if pos_i == constraint_j:
                    # This position's value IS the count for this constraint (special relationship)
                    participation_weight = 1.0
                else:
                    # This position could be counted BY this constraint
                    # Weight by inverse distance for locality preference
                    distance = abs(pos_i - constraint_j)
                    participation_weight = 0.4 + 0.4 * math.exp(-distance / (n/4))
            
            G.add_edge(f'pos_{pos_i}', f'count_constraint_{constraint_j}', 
                      weight=participation_weight)
    
    # Add critical interdependency edges between positions that strongly influence each other
    # In magic sequences, positions with values close to their indices create strong coupling
    for i in range(n):
        for j in range(i + 1, min(i + 5, n)):  # Local neighborhood only
            # Positions are strongly coupled if their expected values are similar
            # or if they're likely to count each other
            distance = j - i
            coupling_strength = 0.2 * math.exp(-distance / 2.0)  # Exponential decay
            
            if coupling_strength > 0.1:  # Only add significant couplings
                G.add_edge(f'pos_{i}', f'pos_{j}', weight=coupling_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2] 
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()