#!/usr/bin/env python3
"""
Graph converter for Nonogram problem.
Created using subagent_prompt.md version: v_02

This problem is about solving nonogram puzzles (Paint by Numbers) where each cell
in a grid can be filled (1) or empty (2) based on row and column constraints.
Key challenges: constraint propagation, backtracking efficiency, constraint tightness.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the nonogram problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with cells (type 0) and constraints (type 1).
    - Cell nodes represent grid positions that need to be filled/empty
    - Row and column constraint nodes represent the constraints
    - Edge weights reflect constraint tightness and cell criticality
    - Harder instances have tighter constraints and more complex patterns
    """
    # Access data directly from json_data dict
    X = json_data.get('X', 0)  # grid width (columns)
    Y = json_data.get('Y', 0)  # grid height (rows)
    maxlen = json_data.get('maxlen', 0)
    rows_flat = json_data.get('rows', [])
    cols_flat = json_data.get('cols', [])
    
    if X == 0 or Y == 0:
        return nx.Graph()
    
    # Reshape flattened arrays to 2D
    rows = []
    for i in range(Y):
        row_constraints = []
        for j in range(maxlen):
            idx = i * maxlen + j
            if idx < len(rows_flat) and rows_flat[idx] >= 0:
                row_constraints.append(rows_flat[idx])
        rows.append(row_constraints)
    
    cols = []
    for i in range(X):
        col_constraints = []
        for j in range(maxlen):
            idx = i * maxlen + j
            if idx < len(cols_flat) and cols_flat[idx] >= 0:
                col_constraints.append(cols_flat[idx])
        cols.append(col_constraints)
    
    # Create graph
    G = nx.Graph()
    
    # Add cell nodes (type 0) - decision variables
    # Weight by centrality and constraint involvement
    for r in range(Y):
        for c in range(X):
            # Central cells tend to be more constrained
            centrality = 1.0 - (abs(r - Y//2) + abs(c - X//2)) / (Y + X)
            
            # Count constraint segments affecting this cell
            row_segments = len([x for x in rows[r] if x > 0])
            col_segments = len([x for x in cols[c] if x > 0])
            constraint_density = (row_segments + col_segments) / 10.0  # normalize
            
            # Combine centrality and constraint density with non-linear weighting
            weight = 0.3 * centrality + 0.7 * min(constraint_density, 1.0)
            weight = math.sqrt(weight)  # Non-linear scaling for better sensitivity
            
            G.add_node(f'cell_{r}_{c}', type=0, weight=weight)
    
    # Add row constraint nodes (type 1)
    for r in range(Y):
        if not rows[r]:  # Empty row constraint
            G.add_node(f'row_{r}', type=1, weight=0.1)
            continue
            
        # Calculate constraint tightness
        total_filled = sum(rows[r])
        total_gaps = len(rows[r]) - 1  # minimum gaps between segments
        min_required = total_filled + total_gaps
        
        if X > 0:
            tightness = min_required / X
            # Add complexity factor for multiple segments
            segment_complexity = len(rows[r]) / 5.0  # normalize by typical max segments
            weight = 0.6 * tightness + 0.4 * segment_complexity
            weight = min(1.0 - math.exp(-3.0 * weight), 1.0)  # Non-linear scaling
        else:
            weight = 0.5
            
        G.add_node(f'row_{r}', type=1, weight=weight)
    
    # Add column constraint nodes (type 1)  
    for c in range(X):
        if not cols[c]:  # Empty column constraint
            G.add_node(f'col_{c}', type=1, weight=0.1)
            continue
            
        # Calculate constraint tightness
        total_filled = sum(cols[c])
        total_gaps = len(cols[c]) - 1
        min_required = total_filled + total_gaps
        
        if Y > 0:
            tightness = min_required / Y
            segment_complexity = len(cols[c]) / 5.0
            weight = 0.6 * tightness + 0.4 * segment_complexity
            weight = min(1.0 - math.exp(-3.0 * weight), 1.0)
        else:
            weight = 0.5
            
        G.add_node(f'col_{c}', type=1, weight=weight)
    
    # Add bipartite edges: cells to their constraining row/column
    max_total_constraint = max(
        max([sum(rows[r]) for r in range(Y)] + [1]),
        max([sum(cols[c]) for c in range(X)] + [1])
    )
    
    for r in range(Y):
        for c in range(X):
            cell = f'cell_{r}_{c}'
            
            # Edge to row constraint
            if rows[r]:
                row_impact = sum(rows[r]) / max_total_constraint
                # Higher weight for cells in tighter constraints
                row_weight = 0.5 + 0.5 * row_impact
            else:
                row_weight = 0.3  # Lower weight for unconstrained rows
                
            G.add_edge(cell, f'row_{r}', weight=row_weight)
            
            # Edge to column constraint
            if cols[c]:
                col_impact = sum(cols[c]) / max_total_constraint
                col_weight = 0.5 + 0.5 * col_impact
            else:
                col_weight = 0.3
                
            G.add_edge(cell, f'col_{c}', weight=col_weight)
    
    # Add conflict edges between highly constrained adjacent cells
    # This captures local difficulty in solving
    for r in range(Y):
        for c in range(X):
            cell_weight = G.nodes[f'cell_{r}_{c}']['weight']
            
            # Only add conflicts for highly constrained cells
            if cell_weight > 0.7:
                # Check adjacent cells
                for dr, dc in [(0, 1), (1, 0)]:  # right and down
                    nr, nc = r + dr, c + dc
                    if nr < Y and nc < X:
                        neighbor = f'cell_{nr}_{nc}'
                        neighbor_weight = G.nodes[neighbor]['weight']
                        
                        # Add conflict edge if both cells are highly constrained
                        if neighbor_weight > 0.7:
                            conflict_strength = (cell_weight + neighbor_weight) / 2.0
                            G.add_edge(f'cell_{r}_{c}', neighbor, weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()