#!/usr/bin/env python3
"""
Graph converter for noseasonal problem.
Created using subagent_prompt.md version: v_02

This problem is about handball league scheduling with complex temporal and venue constraints.
A 14-team league split into two 7-team divisions must schedule matches over 20 periods
with rules about home/away patterns, venue unavailabilities, complementary schedules, and breaks.

Key challenges: 
- Complex home/away pattern constraints (HAP) enforced by finite automaton
- Venue unavailability creates scheduling conflicts
- Complementary schedule requirements between team pairs
- Break minimization while maintaining fairness
- Derby game constraints at specific periods
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the handball scheduling problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling teams, periods, and constraints
    - Team nodes (type 0): 14 teams with weights based on venue restrictions
    - Period nodes (type 2): 20 time periods as limited resources
    - Constraint nodes (type 1): Various scheduling constraints
    - Derby constraint nodes for specific team groupings
    - Venue availability constraints
    - Division assignment constraints
    """
    # Extract data from JSON
    nohome = json_data.get('nohome', [])
    group1 = json_data.get('group1', [])
    group2 = json_data.get('group2', [])
    derby_period = json_data.get('derby_period', [])
    
    # Fixed problem parameters
    num_teams = 14
    num_periods = 20
    divsize = 7
    
    G = nx.Graph()
    
    # Calculate team venue restriction weights
    # Teams with more venue restrictions are more constrained
    if nohome:
        # nohome is flattened - reshape to 14x33 matrix
        venues_per_period = 33
        nohome_matrix = []
        for i in range(num_teams):
            start_idx = i * venues_per_period
            end_idx = start_idx + venues_per_period
            team_restrictions = nohome[start_idx:end_idx] if start_idx < len(nohome) else [0] * venues_per_period
            nohome_matrix.append(team_restrictions)
        
        max_restrictions = max(sum(team_restrictions) for team_restrictions in nohome_matrix)
        max_restrictions = max(max_restrictions, 1)  # Avoid division by zero
    else:
        nohome_matrix = [[0] * 33 for _ in range(num_teams)]
        max_restrictions = 1
    
    # Team nodes (type 0) - weighted by venue restriction severity
    for team in range(1, num_teams + 1):
        restrictions = sum(nohome_matrix[team - 1]) if team <= len(nohome_matrix) else 0
        # More restricted teams have higher weights (more constrained)
        restriction_weight = restrictions / max_restrictions
        # Use exponential scaling to emphasize heavily restricted teams
        venue_weight = math.exp(restriction_weight * 2) / math.exp(2)
        G.add_node(f'team_{team}', type=0, weight=venue_weight)
    
    # Period nodes (type 2) - resources with varying scarcity
    for period in range(1, num_periods + 1):
        # Derby periods are more constrained (limited availability)
        is_derby = period in derby_period
        # First half periods are more constrained due to divisional play
        is_first_half = period <= divsize
        
        scarcity = 0.3  # Base scarcity
        if is_derby:
            scarcity += 0.4
        if is_first_half:
            scarcity += 0.3
        
        G.add_node(f'period_{period}', type=2, weight=min(scarcity, 1.0))
    
    # Division constraint nodes (type 1)
    # Each division has internal constraints
    G.add_node('division1_constraint', type=1, weight=0.8)
    G.add_node('division2_constraint', type=1, weight=0.8)
    
    # Connect teams to their division constraints
    for team in group1:
        if team <= num_teams:
            G.add_edge(f'team_{team}', 'division1_constraint', weight=0.9)
    
    for team in group2:
        if team <= num_teams:
            G.add_edge(f'team_{team}', 'division2_constraint', weight=0.9)
    
    # Derby constraints (type 1) - high weight due to strict requirements
    for i, derby_per in enumerate(derby_period):
        derby_node = f'derby_constraint_{i}'
        G.add_node(derby_node, type=1, weight=0.95)
        
        # Connect derby constraint to its specific period
        if derby_per <= num_periods:
            G.add_edge(derby_node, f'period_{derby_per}', weight=1.0)
    
    # Venue unavailability constraints (type 1)
    for team in range(1, num_teams + 1):
        if team <= len(nohome_matrix):
            restrictions = sum(nohome_matrix[team - 1])
            if restrictions > 0:
                # Create constraint node for teams with venue restrictions
                venue_constraint = f'venue_constraint_team_{team}'
                tightness = min(restrictions / 10.0, 1.0)  # Normalize based on typical restriction counts
                G.add_node(venue_constraint, type=1, weight=tightness)
                
                # Connect team to its venue constraint
                G.add_edge(f'team_{team}', venue_constraint, weight=tightness)
    
    # HAP (Home-Away Pattern) constraints (type 1)
    # The finite automaton creates complex pattern constraints
    G.add_node('hap_pattern_constraint', type=1, weight=0.9)
    
    # Connect all teams to HAP constraint since all must follow the pattern
    for team in range(1, num_teams + 1):
        G.add_edge(f'team_{team}', 'hap_pattern_constraint', weight=0.7)
    
    # Break minimization constraint (type 1)
    # Teams must have exactly 6 aligned breaks per division
    G.add_node('break_constraint_div1', type=1, weight=0.8)
    G.add_node('break_constraint_div2', type=1, weight=0.8)
    
    # Connect division teams to their break constraints
    for team in group1:
        if team <= num_teams:
            G.add_edge(f'team_{team}', 'break_constraint_div1', weight=0.6)
    
    for team in group2:
        if team <= num_teams:
            G.add_edge(f'team_{team}', 'break_constraint_div2', weight=0.6)
    
    # Complementary schedule constraints (type 1)
    # Teams 2,4,6,9,11,13 must have complementary schedules with adjacent teams
    complementary_teams = [2, 4, 6, 9, 11, 13]
    for team in complementary_teams:
        if team <= num_teams:
            comp_constraint = f'complementary_constraint_team_{team}'
            G.add_node(comp_constraint, type=1, weight=0.85)
            G.add_edge(f'team_{team}', comp_constraint, weight=0.9)
            
            # Connect to adjacent team if it exists
            adjacent_team = team + 1 if team < num_teams else team - 1
            if adjacent_team >= 1 and adjacent_team <= num_teams:
                G.add_edge(f'team_{adjacent_team}', comp_constraint, weight=0.9)
    
    # AVR (Alternating Venue Rule) constraint (type 1)
    # Consecutive meetings between teams must be at different venues
    G.add_node('avr_constraint', type=1, weight=0.7)
    
    # Connect all teams to AVR constraint
    for team in range(1, num_teams + 1):
        G.add_edge(f'team_{team}', 'avr_constraint', weight=0.5)
    
    # Round Robin Tournament constraints (type 1)
    G.add_node('rrt_first_constraint', type=1, weight=0.8)
    G.add_node('rrt_second_constraint', type=1, weight=0.8)
    
    # Connect teams to RRT constraints
    for team in range(1, num_teams + 1):
        G.add_edge(f'team_{team}', 'rrt_first_constraint', weight=0.6)
        G.add_edge(f'team_{team}', 'rrt_second_constraint', weight=0.6)
    
    # Connect periods to RRT constraints
    for period in range(1, divsize + 1):  # First tour periods
        G.add_edge(f'period_{period}', 'rrt_first_constraint', weight=0.7)
    
    for period in range(divsize + 1, num_periods + 1):  # Second tour periods
        G.add_edge(f'period_{period}', 'rrt_second_constraint', weight=0.7)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()