#!/usr/bin/env python3
"""
Graph converter for oc-roster problem.
Created using subagent_prompt.md version: v_02

This problem is about staff rostering for on-call duties over weekdays and weekends.
Key challenges: balancing workload, avoiding consecutive assignments, handling availability constraints.
"""

import sys
import json
import math
import networkx as nx
import re
from pathlib import Path


def parse_dzn_sets(dzn_file):
    """Parse unavailable and fixed sets from DZN file since they're not in JSON."""
    unavailable = []
    fixed = []
    
    with open(dzn_file, 'r') as f:
        content = f.read()
    
    # Parse unavailable array
    unavailable_match = re.search(r'unavailable\s*=\s*\[(.*?)\];', content, re.DOTALL)
    if unavailable_match:
        unavailable_str = unavailable_match.group(1)
        # Parse sets like {1, 10, 11}, {}, {3}, etc.
        sets = re.findall(r'\{([^}]*)\}', unavailable_str)
        for s in sets:
            if s.strip():
                unavailable.append([int(x.strip()) for x in s.split(',') if x.strip()])
            else:
                unavailable.append([])
    
    # Parse fixed array  
    fixed_match = re.search(r'fixed\s*=\s*\[(.*?)\];', content, re.DOTALL)
    if fixed_match:
        fixed_str = fixed_match.group(1)
        sets = re.findall(r'\{([^}]*)\}', fixed_str)
        for s in sets:
            if s.strip():
                fixed.append([int(x.strip()) for x in s.split(',') if x.strip()])
            else:
                fixed.append([])
    
    return unavailable, fixed


def calculate_weekend_days(num_days, weekend_offset):
    """Calculate which days are weekends based on the offset."""
    weekend_days = set()
    d = 0
    while d * 5 + weekend_offset + 1 <= num_days:
        weekend_days.add(d * 5 + weekend_offset + 1)
        d += 1
    return weekend_days


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the on-call rostering problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with staff (type 0) and constraints (type 1).
    - Staff nodes weighted by workload and availability constraints
    - Day constraint nodes representing coverage requirements 
    - Workload balance constraints between staff pairs
    - Consecutive day penalty constraints
    - Weekend-before-Wednesday constraints
    """
    # Get basic parameters
    num_staff = json_data.get('num_staff', 0)
    num_days = json_data.get('num_days', 0) 
    weekend_offset = json_data.get('weekend_offset', 0)
    work_load = json_data.get('work_load', [])
    adj_days_str = json_data.get('adj_days_str', 1)
    wed_before_weekend_str = json_data.get('wed_before_weekend_str', 1)
    
    if num_staff == 0 or num_days == 0:
        return nx.Graph()
    
    # Parse additional data from DZN file since it's not in JSON
    dzn_file = str(mzn_file).replace('.mzn', '.dzn')
    if 'converter.py' in str(mzn_file):
        # When testing standalone, extract from sys.argv
        if len(sys.argv) >= 3:
            dzn_file = sys.argv[2]
        else:
            dzn_file = None
    
    unavailable = []
    fixed = []
    if dzn_file and Path(dzn_file).exists():
        unavailable, fixed = parse_dzn_sets(dzn_file)
    
    # Ensure we have data for all staff
    while len(unavailable) < num_staff:
        unavailable.append([])
    while len(fixed) < num_staff:
        fixed.append([])
    while len(work_load) < num_staff:
        work_load.append(100)
    
    # Calculate weekend days
    weekend_days = calculate_weekend_days(num_days, weekend_offset)
    weekdays = set(range(1, num_days + 1)) - weekend_days
    
    G = nx.Graph()
    
    # Staff nodes (type 0) - weighted by availability constraints and workload variation
    max_workload = max(work_load) if work_load else 100
    for s in range(num_staff):
        # Weight combines workload pressure and availability constraints
        unavail_penalty = len(unavailable[s]) / num_days if s < len(unavailable) else 0
        fixed_bonus = len(fixed[s]) / num_days if s < len(fixed) else 0
        workload_factor = work_load[s] / max_workload if s < len(work_load) else 1.0
        
        # Higher weight = more constrained/important staff member
        weight = (workload_factor + unavail_penalty + fixed_bonus) / 3.0
        weight = min(max(weight, 0.1), 1.0)  # Clamp to [0.1, 1.0]
        
        G.add_node(f'staff_{s+1}', type=0, weight=weight)
    
    # Day coverage constraint nodes (type 1)
    for day in range(1, num_days + 1):
        # Weekend days are more constrained (harder to staff)
        is_weekend = day in weekend_days
        weight = 0.8 if is_weekend else 0.4
        G.add_node(f'day_{day}', type=1, weight=weight)
    
    # Workload balance constraint nodes (type 1)
    # Each staff pair needs balanced weekday and weekend assignments
    for i in range(num_staff):
        for j in range(i + 1, num_staff):
            # Tighter constraints when workloads differ significantly
            workload_diff = abs(work_load[i] - work_load[j]) / max_workload
            weight = min(0.3 + workload_diff * 0.5, 1.0)
            
            G.add_node(f'balance_weekday_{i+1}_{j+1}', type=1, weight=weight)
            G.add_node(f'balance_weekend_{i+1}_{j+1}', type=1, weight=weight)
    
    # Consecutive day penalty constraint nodes (type 1)
    penalty_strength = adj_days_str / 5.0  # Normalize penalty strength
    for day in range(1, num_days):
        G.add_node(f'consecutive_{day}_{day+1}', type=1, weight=penalty_strength)
    
    # No-three-consecutive constraint nodes (type 1) - harder constraints
    for day in range(1, num_days - 1):
        G.add_node(f'no_three_consec_{day}_{day+1}_{day+2}', type=1, weight=0.9)
    
    # Wednesday-before-weekend constraint nodes (type 1)
    wed_penalty = wed_before_weekend_str / 5.0
    for weekend_day in weekend_days:
        if weekend_day > 2:  # Has a Wednesday before
            wed_day = weekend_day - 2
            G.add_node(f'wed_before_weekend_{wed_day}_{weekend_day}', type=1, weight=wed_penalty)
    
    # Edges: Staff participation in constraints
    
    # Day coverage edges - staff can be assigned to days they're available
    for s in range(num_staff):
        staff_unavail = set(unavailable[s]) if s < len(unavailable) else set()
        staff_fixed = set(fixed[s]) if s < len(fixed) else set()
        
        for day in range(1, num_days + 1):
            if day not in staff_unavail:
                # Fixed assignments have weight 1.0, others based on preference
                if day in staff_fixed:
                    weight = 1.0
                elif day in weekend_days:
                    weight = 0.7  # Weekend assignments are harder
                else:
                    weight = 0.5
                    
                G.add_edge(f'staff_{s+1}', f'day_{day}', weight=weight)
    
    # Workload balance edges
    for i in range(num_staff):
        for j in range(i + 1, num_staff):
            # Connect both staff to their balance constraints
            balance_strength = 0.6
            G.add_edge(f'staff_{i+1}', f'balance_weekday_{i+1}_{j+1}', weight=balance_strength)
            G.add_edge(f'staff_{j+1}', f'balance_weekday_{i+1}_{j+1}', weight=balance_strength)
            G.add_edge(f'staff_{i+1}', f'balance_weekend_{i+1}_{j+1}', weight=balance_strength)
            G.add_edge(f'staff_{j+1}', f'balance_weekend_{i+1}_{j+1}', weight=balance_strength)
    
    # Consecutive day constraint edges - connect staff to consecutive day penalties
    for s in range(num_staff):
        for day in range(1, num_days):
            staff_unavail = set(unavailable[s]) if s < len(unavailable) else set()
            # Only if both days are available
            if day not in staff_unavail and (day + 1) not in staff_unavail:
                weight = 0.4  # Moderate penalty connection
                G.add_edge(f'staff_{s+1}', f'consecutive_{day}_{day+1}', weight=weight)
    
    # Three-consecutive constraint edges
    for s in range(num_staff):
        for day in range(1, num_days - 1):
            staff_unavail = set(unavailable[s]) if s < len(unavailable) else set()
            staff_fixed = set(fixed[s]) if s < len(fixed) else set()
            
            # Check if all three days are available
            days_available = all(d not in staff_unavail for d in [day, day+1, day+2])
            # Check if all three days are fixed (then constraint doesn't apply)
            all_fixed = all(d in staff_fixed for d in [day, day+1, day+2])
            
            if days_available and not all_fixed:
                weight = 0.8  # Strong constraint
                G.add_edge(f'staff_{s+1}', f'no_three_consec_{day}_{day+1}_{day+2}', weight=weight)
    
    # Wednesday-before-weekend constraint edges
    for s in range(num_staff):
        staff_unavail = set(unavailable[s]) if s < len(unavailable) else set()
        for weekend_day in weekend_days:
            if weekend_day > 2:
                wed_day = weekend_day - 2
                # If both Wednesday and weekend are available
                if wed_day not in staff_unavail and weekend_day not in staff_unavail:
                    weight = 0.3  # Soft preference
                    G.add_edge(f'staff_{s+1}', f'wed_before_weekend_{wed_day}_{weekend_day}', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()