#!/usr/bin/env python3
"""
Graph converter for On-Call Rostering problem.
Created using subagent_prompt.md version: v_02

This problem is about assigning staff members to on-call duties over a time period.
Key challenges: Work load balancing, availability constraints, consecutive day restrictions,
and weekend/weekday fairness across different workload requirements.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the on-call rostering problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model staff-day assignments with workload and temporal constraints
    - Staff nodes (type 0): Weighted by workload demand and availability
    - Day nodes (type 2): Weighted by weekend vs weekday and position criticality
    - Constraint nodes (type 1): Model workload balance, consecutive restrictions
    - Edges model assignment possibilities and temporal relationships
    """
    # Access data from JSON
    num_staff = json_data.get('num_staff', 2)
    num_days = json_data.get('num_days', 7)
    work_load = json_data.get('work_load', [100] * num_staff)
    weekend_offset = json_data.get('weekend_offset', 0)
    adj_days_str = json_data.get('adj_days_str', 1)
    wed_before_weekend_str = json_data.get('wed_before_weekend_str', 1)
    
    # Calculate weekend days based on the problem's weekend model
    weekend_days = set()
    for d in range(num_days // 5 + 1):
        weekend_day = d * 5 + weekend_offset + 1
        if weekend_day <= num_days:
            weekend_days.add(weekend_day)
    
    week_days = set(range(1, num_days + 1)) - weekend_days
    
    G = nx.Graph()
    
    # Staff nodes (type 0) - weighted by workload demand
    max_workload = max(work_load) if work_load else 100
    for s in range(num_staff):
        # Higher workload = higher weight (more demanding/important)
        workload_weight = work_load[s] / max_workload if s < len(work_load) else 0.5
        G.add_node(f'staff_{s}', type=0, weight=workload_weight)
    
    # Day nodes (type 2) - weighted by weekend importance and centrality
    for d in range(1, num_days + 1):
        if d in weekend_days:
            # Weekends are more critical (higher weight)
            weight = 0.8
        else:
            # Weekdays weighted by position - middle days more constrained
            centrality = 1.0 - abs(d - num_days/2) / (num_days/2)
            weight = 0.3 + 0.4 * centrality
        G.add_node(f'day_{d}', type=2, weight=weight)
    
    # Workload balance constraint nodes (type 1)
    # One for weekday balance, one for weekend balance
    weekday_imbalance = len(week_days) / num_staff if num_staff > 0 else 1
    weekend_imbalance = len(weekend_days) / num_staff if num_staff > 0 else 1
    
    G.add_node('weekday_balance', type=1, weight=min(weekday_imbalance / 10, 1.0))
    G.add_node('weekend_balance', type=1, weight=min(weekend_imbalance / 5, 1.0))
    
    # Consecutive day constraint nodes (type 1)
    consecutive_pressure = min(adj_days_str * num_days / (num_staff * 20), 1.0)
    G.add_node('consecutive_constraint', type=1, weight=consecutive_pressure)
    
    # Wednesday-weekend constraint node (type 1)
    wed_weekend_pressure = min(wed_before_weekend_str * len(weekend_days) / 10, 1.0)
    G.add_node('wed_weekend_constraint', type=1, weight=wed_weekend_pressure)
    
    # Staff-day assignment edges (bipartite structure)
    for s in range(num_staff):
        for d in range(1, num_days + 1):
            # Edge weight based on workload compatibility and day importance
            staff_weight = work_load[s] / max_workload if s < len(work_load) else 0.5
            day_weight = 0.8 if d in weekend_days else 0.5
            
            # Higher workload staff more suitable for important days
            assignment_weight = (staff_weight * day_weight + 0.2) / 1.2
            G.add_edge(f'staff_{s}', f'day_{d}', weight=assignment_weight)
    
    # Staff to balance constraint edges
    for s in range(num_staff):
        staff_workload = work_load[s] / max_workload if s < len(work_load) else 0.5
        
        # Higher workload staff more critical for balance
        G.add_edge(f'staff_{s}', 'weekday_balance', weight=staff_workload)
        G.add_edge(f'staff_{s}', 'weekend_balance', weight=staff_workload)
        G.add_edge(f'staff_{s}', 'consecutive_constraint', weight=staff_workload)
    
    # Temporal constraint edges between consecutive days
    for d in range(1, num_days):
        # Stronger weights for critical transitions (weekend boundaries)
        if d in weekend_days or (d + 1) in weekend_days:
            temporal_weight = 0.9
        else:
            temporal_weight = 0.6
        
        G.add_edge(f'day_{d}', f'day_{d+1}', weight=temporal_weight)
        
        # Connect consecutive days to consecutive constraint
        G.add_edge(f'day_{d}', 'consecutive_constraint', weight=temporal_weight * 0.7)
    
    # Wednesday-weekend constraint connections
    for weekend_day in weekend_days:
        if weekend_day > 2:  # Wednesday exists before this weekend
            wed_day = weekend_day - 2
            if wed_day >= 1:
                # Strong connection between Wednesday and following weekend
                weight = 0.8 * wed_before_weekend_str
                G.add_edge(f'day_{wed_day}', f'day_{weekend_day}', weight=weight)
                G.add_edge(f'day_{wed_day}', 'wed_weekend_constraint', weight=weight * 0.8)
                G.add_edge(f'day_{weekend_day}', 'wed_weekend_constraint', weight=weight * 0.8)
    
    # Weekend separation constraint edges
    weekend_list = sorted(list(weekend_days))
    for i in range(len(weekend_list) - 1):
        # Consecutive weekends should not have same staff
        separation_weight = 0.7
        G.add_edge(f'day_{weekend_list[i]}', f'day_{weekend_list[i+1]}', weight=separation_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()