#!/usr/bin/env python3
"""
Graph converter for OOCSP racks problem.
Created using subagent_prompt.md version: v_02

This problem is about configuring rack systems with elements, modules, and frames.
Key challenges: Object cardinality constraints, complex associations between classes,
and strict hierarchical structure (elements need specific modules, modules in frames, frames in racks).
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the OOCSP racks problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the hierarchical object structure and cardinality constraints
    - Class nodes (type 1) represent object type constraints with cardinality weights
    - Association nodes (type 1) represent relationship constraints between classes
    - Virtual object nodes (type 0) represent potential objects with demand-based weights
    - Complex constraint nodes (type 1) for special rules like ModuleII->ModuleV requirements
    """
    
    # Extract instance parameters
    max_objects = json_data.get('MAXNROFOBJECTS', 30)
    use_cardinality = json_data.get('USECARDINALITYCONSTRAINTS', True)
    
    # Class definitions with their bounds
    classes = {
        'Configuration': {'min': json_data.get('CLASS_Configuration_MIN', 0), 'max': 1},
        'Element': {'min': json_data.get('CLASS_Element_MIN', 0), 'max': json_data.get('CLASS_Element_MAX', max_objects)},
        'ElementA': {'min': json_data.get('CLASS_ElementA_MIN', 0), 'max': json_data.get('CLASS_ElementA_MAX', max_objects)},
        'ElementB': {'min': json_data.get('CLASS_ElementB_MIN', 0), 'max': json_data.get('CLASS_ElementB_MAX', max_objects)},
        'ElementC': {'min': json_data.get('CLASS_ElementC_MIN', 0), 'max': json_data.get('CLASS_ElementC_MAX', max_objects)},
        'ElementD': {'min': json_data.get('CLASS_ElementD_MIN', 0), 'max': json_data.get('CLASS_ElementD_MAX', max_objects)},
        'Frame': {'min': json_data.get('CLASS_Frame_MIN', 0), 'max': json_data.get('CLASS_Frame_MAX', max_objects)},
        'Module': {'min': json_data.get('CLASS_Module_MIN', 0), 'max': json_data.get('CLASS_Module_MAX', max_objects)},
        'ModuleI': {'min': json_data.get('CLASS_ModuleI_MIN', 0), 'max': json_data.get('CLASS_ModuleI_MAX', max_objects)},
        'ModuleII': {'min': json_data.get('CLASS_ModuleII_MIN', 0), 'max': json_data.get('CLASS_ModuleII_MAX', max_objects)},
        'ModuleIII': {'min': json_data.get('CLASS_ModuleIII_MIN', 0), 'max': json_data.get('CLASS_ModuleIII_MAX', max_objects)},
        'ModuleIV': {'min': json_data.get('CLASS_ModuleIV_MIN', 0), 'max': json_data.get('CLASS_ModuleIV_MAX', max_objects)},
        'ModuleV': {'min': json_data.get('CLASS_ModuleV_MIN', 0), 'max': json_data.get('CLASS_ModuleV_MAX', max_objects)},
        'Rack': {'min': json_data.get('CLASS_Rack_MIN', 0), 'max': json_data.get('CLASS_Rack_MAX', max_objects)},
        'RackDouble': {'min': json_data.get('CLASS_RackDouble_MIN', 0), 'max': json_data.get('CLASS_RackDouble_MAX', max_objects)},
        'RackSingle': {'min': json_data.get('CLASS_RackSingle_MIN', 0), 'max': json_data.get('CLASS_RackSingle_MAX', max_objects)}
    }
    
    G = nx.Graph()
    
    # Add class constraint nodes (type 1) - each represents cardinality constraints for a class
    for class_name, bounds in classes.items():
        min_objs = bounds['min']
        max_objs = bounds['max'] if 'max' in bounds else max_objects
        
        # Calculate constraint tightness based on minimum requirements vs maximum capacity
        if max_objs > 0:
            demand_ratio = min_objs / max_objs
            # Use non-linear scaling - higher requirements create more constrained nodes
            tightness = 1.0 - math.exp(-3.0 * demand_ratio)
        else:
            tightness = 0.5
            
        G.add_node(f'class_{class_name}', type=1, weight=tightness)
    
    # Add virtual object nodes (type 0) for potential instances
    # These represent the decision space - how many objects of each type to create
    for class_name, bounds in classes.items():
        min_objs = bounds['min']
        max_objs = bounds['max'] if 'max' in bounds else max_objects
        
        if max_objs > 0:
            # Calculate importance based on minimum requirements
            importance = min(min_objs / max(max_objects, 1), 1.0)
            # Use sqrt for non-linear scaling - required objects get higher weight
            obj_weight = math.sqrt(importance) if importance > 0 else 0.1
            
            G.add_node(f'obj_{class_name}', type=0, weight=obj_weight)
            
            # Connect objects to their class constraints
            G.add_edge(f'obj_{class_name}', f'class_{class_name}', weight=0.9)
    
    # Add association constraint nodes (type 1) for key relationships
    
    # Element-Configuration association (elements belong to configuration)
    G.add_node('assoc_Element_Configuration', type=1, weight=0.8)
    if 'obj_Element' in G and 'obj_Configuration' in G:
        G.add_edge('obj_Element', 'assoc_Element_Configuration', weight=0.7)
        G.add_edge('obj_Configuration', 'assoc_Element_Configuration', weight=0.7)
    
    # Frame-Rack association (frames belong to racks)
    G.add_node('assoc_Frame_Rack', type=1, weight=0.8)
    if 'obj_Frame' in G and 'obj_Rack' in G:
        G.add_edge('obj_Frame', 'assoc_Frame_Rack', weight=0.7)
        G.add_edge('obj_Rack', 'assoc_Frame_Rack', weight=0.7)
    
    # Module-Frame association (modules go in frames, max 6 per frame)
    G.add_node('assoc_Module_Frame', type=1, weight=0.9)  # Higher weight - capacity constraint
    if 'obj_Module' in G and 'obj_Frame' in G:
        G.add_edge('obj_Module', 'assoc_Module_Frame', weight=0.8)
        G.add_edge('obj_Frame', 'assoc_Module_Frame', weight=0.8)
    
    # Module-Element association (elements require specific modules)
    G.add_node('assoc_Module_Element', type=1, weight=0.85)
    if 'obj_Module' in G and 'obj_Element' in G:
        G.add_edge('obj_Module', 'assoc_Module_Element', weight=0.8)
        G.add_edge('obj_Element', 'assoc_Module_Element', weight=0.8)
    
    # Rack-Configuration association (racks belong to configuration)
    G.add_node('assoc_Rack_Configuration', type=1, weight=0.7)
    if 'obj_Rack' in G and 'obj_Configuration' in G:
        G.add_edge('obj_Rack', 'assoc_Rack_Configuration', weight=0.6)
        G.add_edge('obj_Configuration', 'assoc_Rack_Configuration', weight=0.6)
    
    # Add specific requirement constraints (type 1)
    
    # ElementA requires 1 ModuleI
    G.add_node('req_ElementA_ModuleI', type=1, weight=0.9)
    if 'obj_ElementA' in G and 'obj_ModuleI' in G:
        G.add_edge('obj_ElementA', 'req_ElementA_ModuleI', weight=1.0)
        G.add_edge('obj_ModuleI', 'req_ElementA_ModuleI', weight=1.0)
    
    # ElementB requires 2 ModuleII
    G.add_node('req_ElementB_ModuleII', type=1, weight=0.95)  # Higher weight - needs 2 modules
    if 'obj_ElementB' in G and 'obj_ModuleII' in G:
        G.add_edge('obj_ElementB', 'req_ElementB_ModuleII', weight=1.0)
        G.add_edge('obj_ModuleII', 'req_ElementB_ModuleII', weight=1.0)
    
    # ElementC requires 3 ModuleIII
    G.add_node('req_ElementC_ModuleIII', type=1, weight=0.98)  # Even higher weight - needs 3 modules
    if 'obj_ElementC' in G and 'obj_ModuleIII' in G:
        G.add_edge('obj_ElementC', 'req_ElementC_ModuleIII', weight=1.0)
        G.add_edge('obj_ModuleIII', 'req_ElementC_ModuleIII', weight=1.0)
    
    # ElementD requires 4 ModuleIV
    G.add_node('req_ElementD_ModuleIV', type=1, weight=1.0)  # Highest weight - needs 4 modules
    if 'obj_ElementD' in G and 'obj_ModuleIV' in G:
        G.add_edge('obj_ElementD', 'req_ElementD_ModuleIV', weight=1.0)
        G.add_edge('obj_ModuleIV', 'req_ElementD_ModuleIV', weight=1.0)
    
    # Special constraint: ModuleII requires ModuleV in same frame
    G.add_node('req_ModuleII_ModuleV_SameFrame', type=1, weight=0.9)
    if 'obj_ModuleII' in G and 'obj_ModuleV' in G:
        G.add_edge('obj_ModuleII', 'req_ModuleII_ModuleV_SameFrame', weight=0.9)
        G.add_edge('obj_ModuleV', 'req_ModuleII_ModuleV_SameFrame', weight=0.9)
    
    # Rack capacity constraints
    # RackSingle has exactly 4 frames
    G.add_node('capacity_RackSingle_4Frames', type=1, weight=0.8)
    if 'obj_RackSingle' in G and 'obj_Frame' in G:
        G.add_edge('obj_RackSingle', 'capacity_RackSingle_4Frames', weight=0.8)
        G.add_edge('obj_Frame', 'capacity_RackSingle_4Frames', weight=0.8)
    
    # RackDouble has exactly 8 frames
    G.add_node('capacity_RackDouble_8Frames', type=1, weight=0.85)
    if 'obj_RackDouble' in G and 'obj_Frame' in G:
        G.add_edge('obj_RackDouble', 'capacity_RackDouble_8Frames', weight=0.8)
        G.add_edge('obj_Frame', 'capacity_RackDouble_8Frames', weight=0.8)
    
    # Add complexity based on cardinality constraints usage
    if use_cardinality:
        # Cardinality constraints make the problem more complex
        G.add_node('cardinality_constraints', type=1, weight=0.9)
        # Connect to major class constraints
        for class_name in ['Element', 'Module', 'Frame', 'Rack']:
            if f'class_{class_name}' in G:
                G.add_edge('cardinality_constraints', f'class_{class_name}', weight=0.6)
    
    # Add hierarchical structure edges between related classes
    # Element hierarchy
    for element_type in ['ElementA', 'ElementB', 'ElementC', 'ElementD']:
        if f'obj_{element_type}' in G and 'obj_Element' in G:
            G.add_edge(f'obj_{element_type}', 'obj_Element', weight=0.5)
    
    # Module hierarchy
    for module_type in ['ModuleI', 'ModuleII', 'ModuleIII', 'ModuleIV', 'ModuleV']:
        if f'obj_{module_type}' in G and 'obj_Module' in G:
            G.add_edge(f'obj_{module_type}', 'obj_Module', weight=0.5)
    
    # Rack hierarchy  
    for rack_type in ['RackSingle', 'RackDouble']:
        if f'obj_{rack_type}' in G and 'obj_Rack' in G:
            G.add_edge(f'obj_{rack_type}', 'obj_Rack', weight=0.5)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()