#!/usr/bin/env python3
"""
Graph converter for Open Stacks problem.
# Converter created with subagent_prompt.md v_02

This problem is about scheduling products to minimize the maximum number of open stacks.
A stack is open when a customer has received some but not all of their ordered products.
Key challenges: Complex interaction between product order and customer order patterns.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Open Stacks problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the complex interactions between customers and products
    - Customers are variable nodes (decision points about when they're active)
    - Products are variable nodes (decision points about ordering)
    - Customer-product relationships create constraints
    - Product overlaps between customers create conflicts
    """
    # Extract data
    c = json_data.get('c', 0)  # customers
    p = json_data.get('p', 0)  # products
    orders_flat = json_data.get('orders', [])
    
    # Convert flat orders array to 2D matrix
    orders = [[0 for _ in range(p)] for _ in range(c)]
    for i in range(c):
        for j in range(p):
            if i * p + j < len(orders_flat):
                orders[i][j] = orders_flat[i * p + j]
    
    G = nx.Graph()
    
    # Calculate customer order counts for weighting
    customer_orders = [sum(orders[i]) for i in range(c)]
    max_customer_orders = max(customer_orders) if customer_orders else 1
    
    # Calculate product popularity for weighting
    product_popularity = [sum(orders[i][j] for i in range(c)) for j in range(p)]
    max_product_popularity = max(product_popularity) if product_popularity else 1
    
    # Customer nodes (type 0) - weighted by order complexity
    for i in range(c):
        num_orders = customer_orders[i]
        if num_orders > 0:
            # Weight by order complexity (more orders = higher weight)
            complexity = num_orders / max_customer_orders
            # Use non-linear scaling for better sensitivity
            weight = math.pow(complexity, 0.7)  # Diminishing returns
            G.add_node(f'customer_{i}', type=0, weight=weight)
    
    # Product nodes (type 0) - weighted by popularity and contention
    for j in range(p):
        popularity = product_popularity[j]
        if popularity > 0:
            # Weight by how contentious this product is
            contention = popularity / max_product_popularity
            # Products ordered by many customers are more critical
            weight = math.sqrt(contention)  # Square root for moderate non-linearity
            G.add_node(f'product_{j}', type=0, weight=weight)
    
    # Create constraint nodes for customer-product relationships (type 1)
    constraint_id = 0
    for i in range(c):
        for j in range(p):
            if orders[i][j] == 1:
                # Each customer-product pair creates a constraint
                # Weight by how this affects scheduling complexity
                customer_load = customer_orders[i] / max_customer_orders
                product_contention = product_popularity[j] / max_product_popularity
                
                # Higher weight for complex interactions
                weight = (customer_load + product_contention) / 2
                G.add_node(f'constraint_{constraint_id}', type=1, weight=weight)
                
                # Connect customer and product to this constraint
                G.add_edge(f'customer_{i}', f'constraint_{constraint_id}', weight=0.8)
                G.add_edge(f'product_{j}', f'constraint_{constraint_id}', weight=0.8)
                
                constraint_id += 1
    
    # Add scheduling constraint node (type 1) - represents the global scheduling constraint
    G.add_node('scheduling_constraint', type=1, weight=1.0)
    
    # Connect all products to scheduling constraint
    for j in range(p):
        if product_popularity[j] > 0:
            # Weight by product importance in scheduling
            importance = product_popularity[j] / max_product_popularity
            weight = 0.5 + 0.5 * importance  # Range [0.5, 1.0]
            G.add_edge(f'product_{j}', 'scheduling_constraint', weight=weight)
    
    # Add conflict edges between customers who share many products
    for i1 in range(c):
        for i2 in range(i1 + 1, c):
            if customer_orders[i1] > 0 and customer_orders[i2] > 0:
                # Count shared products
                shared_products = sum(1 for j in range(p) if orders[i1][j] == 1 and orders[i2][j] == 1)
                
                if shared_products > 0:
                    # Calculate conflict intensity
                    total_products = customer_orders[i1] + customer_orders[i2]
                    conflict_ratio = (2 * shared_products) / total_products
                    
                    # Only add edge if significant conflict
                    if conflict_ratio > 0.3:
                        # Use exponential scaling for conflict intensity
                        weight = 1.0 - math.exp(-3.0 * conflict_ratio)
                        G.add_edge(f'customer_{i1}', f'customer_{i2}', weight=min(weight, 1.0))
    
    # Add conflict edges between highly contentious products
    for j1 in range(p):
        for j2 in range(j1 + 1, p):
            if product_popularity[j1] > 1 and product_popularity[j2] > 1:
                # Products that appear together in many customer orders create sequencing conflicts
                shared_customers = sum(1 for i in range(c) if orders[i][j1] == 1 and orders[i][j2] == 1)
                
                if shared_customers > 0:
                    # Calculate sequencing conflict
                    total_customers = product_popularity[j1] + product_popularity[j2]
                    conflict_ratio = (2 * shared_customers) / total_customers
                    
                    # Only add edge if significant conflict
                    if conflict_ratio > 0.4:
                        # Products that must appear in many same customer orders create sequencing pressure
                        weight = math.tanh(2.0 * conflict_ratio)  # Tanh for smooth saturation
                        G.add_edge(f'product_{j1}', f'product_{j2}', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()