#!/usr/bin/env python3
"""
Graph converter for p1f-pjs (Perfect 1-Factorization) problem.
Created using subagent_prompt.md version: v_02

This problem is about finding a perfect 1-factorization of the complete graph K_n.
A 1-factorization partitions the edges of K_n into (n-1) perfect matchings such that
every pair of matchings forms a Hamiltonian circuit.

Key challenges: 
- Complex combinatorial structure with multiple global constraints
- Perfect matchings require specific bipartite structures
- Hamiltonian circuit constraint between matching pairs creates dependency complexity
- Symmetry breaking constraints add structural relationships
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the perfect 1-factorization problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the combinatorial structure of K_n 1-factorization
    - Variable nodes represent matching assignments p[row,col]
    - Constraint nodes model the complex constraint interactions
    - Resource nodes represent the graph vertices being matched
    - Edge weights reflect constraint tightness and matching complexity
    
    Key insights:
    - Problem difficulty grows exponentially with n (only even n have solutions)
    - Central positions in matching matrix are more constrained
    - Matching constraints interact heavily with circuit constraints
    - Symmetry breaking creates ordering dependencies
    """
    # Access data from json_data dict
    n = json_data.get('n', 0)
    m = n - 1  # Number of matchings
    
    # Create graph
    G = nx.Graph()
    
    if n <= 0:
        return G
    
    # Variable nodes: matching assignments p[row,col]
    # Weight by position criticality - edge positions and center are most constrained
    for row in range(1, m + 1):
        for col in range(1, n + 1):
            # Edge positions (first/last) and central positions are more critical
            edge_factor = 1.0 if (row == 1 or row == m or col == 1 or col == n) else 0.7
            center_factor = 1.0 - abs(col - (n + 1) / 2) / (n / 2)
            # Combine factors with exponential weighting for complexity
            criticality = (edge_factor * 0.6 + center_factor * 0.4) * math.exp(-0.1 * (n - 10))
            weight = min(max(criticality, 0.1), 1.0)
            G.add_node(f'p_{row}_{col}', type=0, weight=weight)
    
    # Resource nodes: graph vertices being matched
    # Weight by centrality - more central vertices participate in more constraints
    for v in range(1, n + 1):
        centrality = 1.0 - abs(v - (n + 1) / 2) / (n / 2)
        # Higher weight for vertices that appear in more matching positions
        weight = 0.3 + 0.7 * centrality
        G.add_node(f'vertex_{v}', type=2, weight=weight)
    
    # Constraint nodes modeling the complex constraint structure
    
    # 1. No self-loop constraints (n*m constraints)
    for row in range(1, m + 1):
        for col in range(1, n + 1):
            # Simple constraint, lower weight
            G.add_node(f'no_self_loop_{row}_{col}', type=1, weight=0.2)
    
    # 2. Matching constraints (m inverse constraints, very tight)
    for row in range(1, m + 1):
        # Inverse constraint is complex and tight
        tightness = 0.9  # Very restrictive constraint
        G.add_node(f'matching_{row}', type=1, weight=tightness)
    
    # 3. Partition constraints (n all_different constraints)
    for col in range(1, n + 1):
        # All_different over m variables
        scope_factor = min(m / 10.0, 1.0)  # More variables = more complex
        weight = 0.7 + 0.3 * scope_factor
        G.add_node(f'partition_{col}', type=1, weight=weight)
    
    # 4. Hamiltonian circuit constraints (most complex)
    circuit_count = 0
    for rowa in range(1, m + 1):
        for rowb in range(rowa + 1, m + 1):
            circuit_count += 1
            # Circuit constraint is the most complex - exponential difficulty
            complexity = math.exp(-0.05 * n) + 0.5  # Decreases with n but stays substantial
            weight = min(complexity, 1.0)
            G.add_node(f'circuit_{rowa}_{rowb}', type=1, weight=weight)
    
    # 5. Lexicographic ordering constraints (m-1 constraints)
    for row in range(1, m):
        # Symmetry breaking constraint
        weight = 0.4  # Moderate complexity
        G.add_node(f'lex_order_{row}', type=1, weight=weight)
    
    # 6. Objective constraint
    G.add_node('objective_constraint', type=1, weight=0.3)
    
    # Add bipartite edges: variables participate in constraints
    
    # No self-loop constraint edges
    for row in range(1, m + 1):
        for col in range(1, n + 1):
            var_node = f'p_{row}_{col}'
            constraint_node = f'no_self_loop_{row}_{col}'
            G.add_edge(var_node, constraint_node, weight=0.3)
    
    # Matching constraint edges (each variable in row participates)
    for row in range(1, m + 1):
        constraint_node = f'matching_{row}'
        for col in range(1, n + 1):
            var_node = f'p_{row}_{col}'
            # Higher weight for participation in complex matching constraint
            G.add_edge(var_node, constraint_node, weight=0.8)
    
    # Partition constraint edges (each variable in column participates)
    for col in range(1, n + 1):
        constraint_node = f'partition_{col}'
        for row in range(1, m + 1):
            var_node = f'p_{row}_{col}'
            # Weight by position importance in all_different
            participation_weight = 0.6 + 0.3 * (1.0 - abs(row - m/2) / (m/2))
            G.add_edge(var_node, constraint_node, weight=participation_weight)
    
    # Circuit constraint edges (two rows participate in each circuit)
    for rowa in range(1, m + 1):
        for rowb in range(rowa + 1, m + 1):
            constraint_node = f'circuit_{rowa}_{rowb}'
            # All variables in both rows participate
            for col in range(1, n + 1):
                var_node_a = f'p_{rowa}_{col}'
                var_node_b = f'p_{rowb}_{col}'
                # High weight for circuit participation
                circuit_weight = 0.9
                G.add_edge(var_node_a, constraint_node, weight=circuit_weight)
                G.add_edge(var_node_b, constraint_node, weight=circuit_weight)
    
    # Lexicographic ordering constraint edges
    for row in range(1, m):
        constraint_node = f'lex_order_{row}'
        for col in range(1, n + 1):
            var_node_a = f'p_{row}_{col}'
            var_node_b = f'p_{row+1}_{col}'
            # Weight by position importance in lexicographic ordering
            lex_weight = 0.5 + 0.4 * (1.0 - (col - 1) / n)  # Earlier positions more important
            G.add_edge(var_node_a, constraint_node, weight=lex_weight)
            G.add_edge(var_node_b, constraint_node, weight=lex_weight)
    
    # Objective constraint edges (first row variables)
    for col in range(1, n + 1):
        var_node = f'p_1_{col}'
        # Weight by coefficient importance
        coeff_weight = col / n  # Higher columns have higher coefficients
        G.add_edge(var_node, 'objective_constraint', weight=coeff_weight)
    
    # Resource constraint edges: variables constrained by vertex assignments
    for row in range(1, m + 1):
        for col in range(1, n + 1):
            var_node = f'p_{row}_{col}'
            vertex_node = f'vertex_{col}'
            # Connection strength based on matching complexity
            usage_weight = 0.4 + 0.3 * math.exp(-0.1 * n)
            G.add_edge(var_node, vertex_node, weight=usage_weight)
    
    # Add conflict edges between variables that cannot be equal (same row)
    for row in range(1, m + 1):
        vars_in_row = [f'p_{row}_{col}' for col in range(1, n + 1)]
        for i in range(len(vars_in_row)):
            for j in range(i + 1, min(i + 4, len(vars_in_row))):  # Limit connections to avoid too dense
                # Conflict weight based on matching constraint tightness
                conflict_weight = 0.3 * math.exp(-abs(i - j) / 3.0)  # Closer variables have stronger conflicts
                G.add_edge(vars_in_row[i], vars_in_row[j], weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()