#!/usr/bin/env python3
"""
Graph converter for Perfect 1-Factorization (p1f) problem.
Converter created with subagent_prompt.md v_02

This problem finds a perfect 1-factorization of complete graph K_n.
A 1-factorization partitions the edges into m=n-1 complete matchings,
where each pair of matchings forms a Hamiltonian circuit.

Key challenges: 
- Exponential constraint interactions between matchings
- Complex Hamiltonian circuit requirements between matching pairs  
- Global alldifferent constraints for partitioning
- Symmetry breaking constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the perfect 1-factorization problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the constraint structure
    - Vertex nodes (type 0): Represent positions in the factorization matrix p[i,j]
    - Matching constraint nodes (type 1): Each row must be a perfect matching
    - Partition constraint nodes (type 1): Each column must be all-different  
    - Circuit constraint nodes (type 1): Each pair of rows must form Hamiltonian circuit
    - Edges weight by constraint scope and tightness
    """
    n = json_data.get('n', 0)
    if n <= 0:
        return nx.Graph()
    
    m = n - 1  # Number of matchings
    G = nx.Graph()
    
    # Position nodes (decision variables p[i,j])
    # Weight by centrality - positions that participate in more constraints are more critical
    max_constraint_participation = 2 + (m - 1)  # matching + partition + (m-1) circuits
    for i in range(1, m + 1):  # rows 1..m
        for j in range(1, n + 1):  # cols 1..n
            # Each position participates in: 1 matching + 1 partition + (m-1) circuit constraints
            participation = max_constraint_participation
            centrality = participation / max_constraint_participation
            G.add_node(f'pos_{i}_{j}', type=0, weight=centrality)
    
    # Matching constraint nodes (one per row)
    # Each row must be a perfect matching (inverse constraint)
    for i in range(1, m + 1):
        # Matching constraints become tighter with larger n (more variables to coordinate)
        tightness = 1.0 - math.exp(-n / 8.0)  # Non-linear scaling
        G.add_node(f'matching_{i}', type=1, weight=tightness)
        
        # Connect to all positions in this row
        for j in range(1, n + 1):
            # Weight by position centrality in the matching
            edge_weight = 1.0 / n  # Each position equally important in matching
            G.add_edge(f'pos_{i}_{j}', f'matching_{i}', weight=edge_weight)
    
    # Partition constraint nodes (one per column)  
    # Each column must have all different values (alldifferent)
    for j in range(1, n + 1):
        # Partition constraints get tighter with more matchings
        tightness = 1.0 - math.exp(-m / 6.0)  # Non-linear scaling
        G.add_node(f'partition_{j}', type=1, weight=tightness)
        
        # Connect to all positions in this column
        for i in range(1, m + 1):
            # Weight by position importance in partition
            edge_weight = 1.0 / m  # Each position equally important in partition
            G.add_edge(f'pos_{i}_{j}', f'partition_{j}', weight=edge_weight)
    
    # Circuit constraint nodes (one per pair of rows)
    # Each pair of matchings must form Hamiltonian circuit
    circuit_count = 0
    for i1 in range(1, m + 1):
        for i2 in range(i1 + 1, m + 1):
            circuit_count += 1
            # Circuit constraints are the most complex - highest weight
            # Difficulty increases exponentially with graph size
            complexity = 1.0 - math.exp(-n / 5.0)  # Exponential difficulty growth
            G.add_node(f'circuit_{i1}_{i2}', type=1, weight=complexity)
            
            # Connect to all positions in both rows
            for j in range(1, n + 1):
                # Each position participates equally in the circuit constraint
                circuit_weight = 0.8  # High weight for critical circuit constraints
                G.add_edge(f'pos_{i1}_{j}', f'circuit_{i1}_{i2}', weight=circuit_weight)
                G.add_edge(f'pos_{i2}_{j}', f'circuit_{i1}_{i2}', weight=circuit_weight)
    
    # Add complexity node representing overall problem difficulty
    total_constraints = m + n + (m * (m - 1)) // 2  # matching + partition + circuit
    overall_complexity = 1.0 - math.exp(-total_constraints / 20.0)
    G.add_node('global_complexity', type=1, weight=overall_complexity)
    
    # Connect global complexity to the most constrained positions
    # Positions in middle rows/columns tend to be most constrained
    mid_row = (m + 1) // 2
    mid_col = (n + 1) // 2
    for i in range(max(1, mid_row - 1), min(m + 1, mid_row + 2)):
        for j in range(max(1, mid_col - 1), min(n + 1, mid_col + 2)):
            # Distance from center affects complexity contribution
            distance = abs(i - mid_row) + abs(j - mid_col)
            contribution = math.exp(-distance / 2.0)
            G.add_edge(f'pos_{i}_{j}', 'global_complexity', weight=contribution)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()