#!/usr/bin/env python3
"""
Graph converter for Peaceable Queens problem.
Created using subagent_prompt.md version: v_02

This problem is about placing equal numbers of black and white queens on an n×n 
chessboard such that no queen attacks another queen of the opposite color.
Key challenges: Position selection for non-attacking placement, exponential search space.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Peaceable Queens problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling chess positions and attack constraints
    - Board positions are variables (type 0) - weighted by centrality and attack potential
    - Attack constraints are explicit constraint nodes (type 1) - for all attack patterns
    - Resource constraints for equal number of queens (type 2)
    
    The graph captures the conflict structure and position importance that affects
    solving difficulty in peaceable queens placement.
    """
    n = json_data.get('n', 8)
    
    G = nx.Graph()
    
    # Type 0: Board position nodes (variables for queen placement)
    # Weight positions by their strategic importance and attack potential
    for r in range(n):
        for c in range(n):
            # Calculate position importance factors
            # 1. Distance from center (central positions are more constrained)
            center_dist = abs(r - (n-1)/2) + abs(c - (n-1)/2)
            max_center_dist = 2 * (n-1)/2
            centrality = 1.0 - (center_dist / max_center_dist)
            
            # 2. Attack potential (how many squares this position can attack)
            # Queens can attack entire row, column, and both diagonals
            attack_squares = (n-1) + (n-1)  # row + column (excluding self)
            
            # Diagonal attacks depend on position
            # Main diagonal (top-left to bottom-right)
            main_diag_len = n - abs(r - c)
            # Anti-diagonal (top-right to bottom-left) 
            anti_diag_len = n - abs(r + c - (n-1))
            attack_squares += (main_diag_len - 1) + (anti_diag_len - 1)
            
            # Maximum possible attacks for any position
            max_attacks = 4 * (n - 1)  # Conservative estimate
            attack_ratio = attack_squares / max_attacks
            
            # Combine centrality and attack potential with non-linear weighting
            # Use exponential to emphasize highly strategic positions
            strategic_value = 0.6 * centrality + 0.4 * attack_ratio
            weight = math.exp(-1.0 * (1.0 - strategic_value)) / math.e
            
            G.add_node(f'pos_{r}_{c}', type=0, weight=weight)
    
    # Type 1: Attack constraint nodes for every attack pattern
    # These model the "no attack between colors" constraints
    
    # Row attack constraints - each row can have at most one color attacking pattern
    for r in range(n):
        # Weight by row length (all rows same, but consistent with model)
        G.add_node(f'row_attack_{r}', type=1, weight=1.0)
        
        # Connect all positions in this row
        for c in range(n):
            G.add_edge(f'pos_{r}_{c}', f'row_attack_{r}', weight=1.0)
    
    # Column attack constraints
    for c in range(n):
        G.add_node(f'col_attack_{c}', type=1, weight=1.0)
        
        for r in range(n):
            G.add_edge(f'pos_{r}_{c}', f'col_attack_{c}', weight=1.0)
    
    # Main diagonal attack constraints (top-left to bottom-right)
    for d in range(-(n-1), n):  # Diagonal offset from main diagonal
        positions = []
        for r in range(n):
            c = r + d
            if 0 <= c < n:
                positions.append((r, c))
        
        if len(positions) > 1:  # Only create constraint if diagonal has multiple positions
            diag_weight = len(positions) / n  # Weight by diagonal length
            G.add_node(f'main_diag_{d}', type=1, weight=diag_weight)
            
            for r, c in positions:
                G.add_edge(f'pos_{r}_{c}', f'main_diag_{d}', weight=1.0)
    
    # Anti-diagonal attack constraints (top-right to bottom-left)
    for s in range(0, 2*n-1):  # Sum of coordinates for anti-diagonal
        positions = []
        for r in range(n):
            c = s - r
            if 0 <= c < n:
                positions.append((r, c))
        
        if len(positions) > 1:
            diag_weight = len(positions) / n
            G.add_node(f'anti_diag_{s}', type=1, weight=diag_weight)
            
            for r, c in positions:
                G.add_edge(f'pos_{r}_{c}', f'anti_diag_{s}', weight=1.0)
    
    # Type 2: Resource constraints
    # Equal number constraint - both colors must have same count
    G.add_node('equal_count_constraint', type=2, weight=1.0)
    
    # Connect all positions to the equality constraint
    # Weight by inverse of board size (larger boards have more flexibility)
    equality_weight = 1.0 / math.sqrt(n)
    for r in range(n):
        for c in range(n):
            G.add_edge(f'pos_{r}_{c}', 'equal_count_constraint', weight=equality_weight)
    
    # Add complexity constraints for larger boards
    if n >= 10:
        # Global complexity node representing the exponential nature of the problem
        complexity_weight = math.log(n) / math.log(20)  # Logarithmic scaling
        G.add_node('global_complexity', type=1, weight=complexity_weight)
        
        # Connect to central positions which are most constrained
        center = (n-1) // 2
        for r in range(max(0, center-1), min(n, center+2)):
            for c in range(max(0, center-1), min(n, center+2)):
                complexity_edge_weight = 1.0 - abs(r - center) / n - abs(c - center) / n
                G.add_edge(f'pos_{r}_{c}', 'global_complexity', weight=complexity_edge_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()