#!/usr/bin/env python3
"""
Graph converter for pennies problem.
Created using subagent_prompt.md version: v_02

This problem is about placing n pennies on an n×n chessboard such that all 
pairwise distances between pennies are distinct. This is a geometric constraint 
satisfaction problem with both placement and distance uniqueness constraints.

Key challenges: 
- Exponential growth in number of possible distance pairs (n choose 2)
- Geometric constraints on board positions
- Distance uniqueness becomes harder as board fills up
- Symmetry breaking needed due to penny indistinguishability
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the pennies problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph capturing placement and distance constraints
    - Position nodes (type 0): Each board square is a potential penny location
    - Constraint nodes (type 1): Distance uniqueness and placement constraints
    - Edge weights reflect geometric relationships and constraint criticality
    
    The core difficulty comes from ensuring all pairwise distances are unique
    while maximizing the number of pennies placed.
    """
    n = json_data.get('n', 5)
    
    G = nx.Graph()
    
    # Position nodes (type 0): Each square on the n×n board
    # Weight based on centrality and potential conflicts
    for i in range(n):
        for j in range(n):
            # Central positions are more constrained due to more potential distances
            center_distance = math.sqrt((i - n//2)**2 + (j - n//2)**2)
            max_center_distance = math.sqrt(2) * n//2
            
            # Non-linear weighting: central positions are exponentially more critical
            centrality = math.exp(-2.0 * center_distance / max_center_distance) if max_center_distance > 0 else 1.0
            
            # Corner and edge positions have fewer potential conflicts
            edge_bonus = 0.0
            if i == 0 or i == n-1 or j == 0 or j == n-1:
                edge_bonus = 0.2
            if (i == 0 or i == n-1) and (j == 0 or j == n-1):
                edge_bonus = 0.4  # Corner positions
            
            weight = min(centrality + edge_bonus, 1.0)
            G.add_node(f'pos_{i}_{j}', type=0, weight=weight)
    
    # Distance uniqueness constraints (type 1)
    # Create constraint nodes for distance ranges to model uniqueness pressure
    total_positions = n * n
    max_pennies = n  # Typical maximum based on problem structure
    
    # Group distances by magnitude to model constraint tightness
    distance_ranges = []
    max_dist_squared = 2 * (n-1)**2  # Maximum possible squared distance
    
    # Create distance range constraints
    num_ranges = min(8, n)  # Reasonable number of distance ranges
    for r in range(num_ranges):
        min_dist = (r * max_dist_squared) // num_ranges
        max_dist = ((r + 1) * max_dist_squared) // num_ranges
        
        # Count how many position pairs fall in this distance range
        pairs_in_range = 0
        for i1 in range(n):
            for j1 in range(n):
                for i2 in range(i1, n):
                    for j2 in range(n):
                        if i1 == i2 and j1 >= j2:
                            continue
                        dist_sq = (i1 - i2)**2 + (j1 - j2)**2
                        if min_dist <= dist_sq < max_dist:
                            pairs_in_range += 1
        
        # Constraint tightness: more pairs in range = higher conflict potential
        if pairs_in_range > 0:
            tightness = min(pairs_in_range / (total_positions / 4), 1.0)
            # Use logarithmic scaling for better discrimination
            weight = math.log(1 + tightness * math.e) / math.log(1 + math.e)
            G.add_node(f'dist_range_{r}', type=1, weight=weight)
            distance_ranges.append((r, min_dist, max_dist, pairs_in_range))
    
    # All-different constraint for positions (prevents multiple pennies per square)
    G.add_node('all_diff_positions', type=1, weight=0.9)
    
    # Distance uniqueness global constraint (most critical)
    max_unique_distances = (max_pennies * (max_pennies - 1)) // 2
    available_distances = len(set((i1-i2)**2 + (j1-j2)**2 
                                for i1 in range(n) for j1 in range(n)
                                for i2 in range(n) for j2 in range(n)
                                if i1 != i2 or j1 != j2))
    
    uniqueness_pressure = max_unique_distances / max(available_distances, 1)
    uniqueness_weight = min(uniqueness_pressure, 1.0)
    G.add_node('all_diff_distances', type=1, weight=uniqueness_weight)
    
    # Bipartite edges: position participation in constraints
    
    # Connect positions to distance range constraints
    for i in range(n):
        for j in range(n):
            pos_node = f'pos_{i}_{j}'
            
            # Connect to all-different position constraint
            G.add_edge(pos_node, 'all_diff_positions', weight=1.0)
            
            # Connect to distance uniqueness constraint with participation weight
            # Central positions participate more heavily in distance constraints
            center_dist = math.sqrt((i - n//2)**2 + (j - n//2)**2)
            max_center_dist = math.sqrt(2) * n//2 if n > 1 else 1.0
            participation = max(0.0, 1.0 - (center_dist / max_center_dist))
            G.add_edge(pos_node, 'all_diff_distances', weight=participation)
            
            # Connect to relevant distance range constraints
            for range_id, min_dist, max_dist, pairs_count in distance_ranges:
                # Check how many distances from this position fall in the range
                distances_in_range = 0
                for i2 in range(n):
                    for j2 in range(n):
                        if i == i2 and j == j2:
                            continue
                        dist_sq = (i - i2)**2 + (j - j2)**2
                        if min_dist <= dist_sq < max_dist:
                            distances_in_range += 1
                
                if distances_in_range > 0:
                    # Weight by how much this position contributes to range conflicts
                    range_participation = distances_in_range / max(pairs_count, 1)
                    edge_weight = min(range_participation * 2, 1.0)
                    G.add_edge(pos_node, f'dist_range_{range_id}', weight=edge_weight)
    
    # Add conflict edges between positions that would create identical distances
    # This models the core difficulty of the problem
    for i1 in range(n):
        for j1 in range(n):
            for i2 in range(i1, n):
                for j2 in range(n):
                    if i1 == i2 and j1 >= j2:
                        continue
                    
                    pos1 = f'pos_{i1}_{j1}'
                    pos2 = f'pos_{i2}_{j2}'
                    
                    # Find if there are other position pairs with same distance
                    dist_sq = (i1 - i2)**2 + (j1 - j2)**2
                    conflicting_pairs = 0
                    
                    for i3 in range(n):
                        for j3 in range(n):
                            for i4 in range(i3, n):
                                for j4 in range(n):
                                    if i3 == i4 and j3 >= j4:
                                        continue
                                    if (i3 == i1 and j3 == j1 and i4 == i2 and j4 == j2):
                                        continue
                                    
                                    other_dist_sq = (i3 - i4)**2 + (j3 - j4)**2
                                    if dist_sq == other_dist_sq:
                                        conflicting_pairs += 1
                    
                    # Add conflict edge if this distance appears multiple times
                    if conflicting_pairs > 0:
                        conflict_strength = min(conflicting_pairs / 10.0, 1.0)
                        # Use exponential scaling for conflict severity
                        conflict_weight = 1.0 - math.exp(-conflict_strength * 3)
                        G.add_edge(pos1, pos2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()