#!/usr/bin/env python3
"""
Graph converter for pentominoes-int problem.
Created using subagent_prompt.md version: v_02

This problem is about placing pentomino tiles on a board to fill a given shape.
Key challenges: spatial constraints, tile orientation/rotation possibilities, and board coverage.
Each tile must be placed exactly once without overlapping.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the pentomino placement problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model board positions as variables and placement constraints as constraint nodes.
    - Board positions are type 0 nodes (decision variables)
    - Each tile placement constraint is a type 1 node 
    - Finite automaton transitions create type 1 constraint nodes
    - Edge weights represent constraint strength and position importance
    """
    
    # Extract problem data
    width = json_data.get('width', 0)
    height = json_data.get('height', 0)
    filled = json_data.get('filled', 1)
    ntiles = json_data.get('ntiles', 0)
    tiles = json_data.get('tiles', [])
    dfa = json_data.get('dfa', [])
    
    G = nx.Graph()
    
    if width == 0 or height == 0:
        return G
    
    board_size = width * height
    
    # Add board position nodes (type 0 - variables)
    # Weight based on position centrality and boundary effects
    for h in range(height):
        for w in range(width):
            pos = h * width + w
            
            # Calculate centrality weight - center positions are more constrained
            center_h, center_w = height / 2, width / 2
            dist_to_center = math.sqrt((h - center_h)**2 + (w - center_w)**2)
            max_dist = math.sqrt(center_h**2 + center_w**2)
            centrality = 1.0 - (dist_to_center / max_dist) if max_dist > 0 else 1.0
            
            # Corner and edge positions have different constraint patterns
            is_corner = (h == 0 or h == height-1) and (w == 0 or w == width-1)
            is_edge = (h == 0 or h == height-1 or w == 0 or w == width-1)
            
            if is_corner:
                weight = 0.9 + 0.1 * centrality  # Corners are highly constrained
            elif is_edge:
                weight = 0.7 + 0.3 * centrality  # Edges somewhat constrained
            else:
                weight = 0.4 + 0.6 * centrality  # Interior positions variable
            
            G.add_node(f'pos_{h}_{w}', type=0, weight=weight)
    
    # Process tiles array - each tile has [q, s, fstart, fend, dstart] format
    tiles_per_entry = 5  # Based on the MZN model structure
    processed_tiles = []
    
    for t in range(ntiles):
        base_idx = t * tiles_per_entry
        if base_idx + 4 < len(tiles):
            q = tiles[base_idx]      # DFA states
            s = tiles[base_idx + 1]  # DFA symbols
            fstart = tiles[base_idx + 2]  # Final states start
            fend = tiles[base_idx + 3]    # Final states end
            dstart = tiles[base_idx + 4]  # DFA transitions start
            
            processed_tiles.append({
                'id': t,
                'q': q, 's': s, 'fstart': fstart, 'fend': fend, 'dstart': dstart
            })
    
    # Add tile constraint nodes (type 1) - one per tile
    max_states = max([tile['q'] for tile in processed_tiles]) if processed_tiles else 1
    for tile in processed_tiles:
        # Weight based on automaton complexity and final state ratio
        state_complexity = tile['q'] / max_states if max_states > 0 else 0.5
        final_states = max(tile['fend'] - tile['fstart'] + 1, 1)
        final_ratio = final_states / max(tile['q'], 1)
        
        # More complex automata (more states, fewer final states) are harder constraints
        constraint_weight = 0.3 + 0.4 * state_complexity + 0.3 * (1.0 - final_ratio)
        
        G.add_node(f'tile_{tile["id"]}', type=1, weight=constraint_weight)
    
    # Add global shape constraint nodes
    # Border constraint - positions at rightmost column must be ntiles+1 (empty)
    G.add_node('border_constraint', type=1, weight=0.8)
    
    # Coverage constraint - ensure all filled positions are covered
    G.add_node('coverage_constraint', type=1, weight=0.9)
    
    # Add edges: position-to-constraint relationships
    for h in range(height):
        for w in range(width):
            pos_node = f'pos_{h}_{w}'
            pos = h * width + w
            
            # Connect border positions to border constraint
            if w == width - 1:  # Rightmost column
                # These positions must be ntiles+1 (empty)
                G.add_edge(pos_node, 'border_constraint', weight=1.0)
            else:
                # These positions participate in coverage
                G.add_edge(pos_node, 'coverage_constraint', weight=0.7)
            
            # Connect positions to tile constraints based on reachability
            # Positions near board edges have different tile placement possibilities
            edge_distance = min(h, height-1-h, w, width-1-w)
            max_edge_dist = min(height//2, width//2)
            
            for tile in processed_tiles:
                # Connection strength based on position accessibility for this tile
                # Tiles with more states can potentially reach more positions
                reach_factor = min(tile['q'] / 10.0, 1.0)  # Normalize state count
                edge_factor = edge_distance / max(max_edge_dist, 1)
                
                connection_strength = 0.4 + 0.3 * reach_factor + 0.3 * edge_factor
                
                # Not all positions connect to all tiles - add selectivity
                if (pos + tile['id']) % 3 == 0:  # Selective connection pattern
                    G.add_edge(pos_node, f'tile_{tile["id"]}', weight=connection_strength)
    
    # Add inter-tile conflict edges for overlapping placements
    # Tiles that have similar automaton patterns might conflict more
    for i, tile1 in enumerate(processed_tiles):
        for j, tile2 in enumerate(processed_tiles[i+1:], i+1):
            # Conflict strength based on automaton similarity
            state_diff = abs(tile1['q'] - tile2['q'])
            symbol_diff = abs(tile1['s'] - tile2['s'])
            max_diff = max(max_states, max([tile['s'] for tile in processed_tiles] + [1]))
            
            if max_diff > 0:
                similarity = 1.0 - (state_diff + symbol_diff) / (2 * max_diff)
                conflict_weight = 0.2 + 0.6 * similarity
                
                # Add conflict edge if similarity is high enough
                if similarity > 0.5:
                    G.add_edge(f'tile_{tile1["id"]}', f'tile_{tile2["id"]}', 
                             weight=conflict_weight)
    
    # Add DFA transition complexity as constraint nodes
    # Group DFA states into constraint clusters
    unique_states = set()
    if len(dfa) > 0:
        # Sample DFA to identify state patterns
        sample_size = min(len(dfa), 1000)
        for i in range(0, sample_size, 10):
            if i < len(dfa):
                unique_states.add(dfa[i])
    
    # Create constraint nodes for major state groups
    for state_group in range(min(len(unique_states), 8)):  # Limit complexity
        state_weight = 0.5 + 0.3 * (state_group / 8.0)  # Higher groups are more complex
        G.add_node(f'dfa_group_{state_group}', type=1, weight=state_weight)
        
        # Connect to relevant positions based on state group
        for h in range(height):
            for w in range(width):
                if (h + w + state_group) % 4 == 0:  # Selective connection
                    connection_weight = 0.3 + 0.2 * (state_group / 8.0)
                    G.add_edge(f'pos_{h}_{w}', f'dfa_group_{state_group}', 
                             weight=connection_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()