#!/usr/bin/env python3
"""
Graph converter for Pentominoes placement problem.
Created using subagent_prompt.md version: v_02

This problem is about placing pentomino pieces on a rectangular board using DFA constraints.
Key challenges: spatial placement conflicts, tile rotation/orientation constraints, 
board boundary constraints, and complex DFA state transitions for valid tile patterns.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the pentominoes placement problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with board positions as variables and
    placement constraints as constraint nodes. Model spatial conflicts and
    DFA complexity through edge weights and node importance.
    - Board positions are type 0 (variables)
    - Tile placement constraints are type 1 (constraints)  
    - Boundary and adjacency constraints are type 1
    - Edge positions and corners are more constrained (higher weight)
    """
    # Extract problem parameters
    width = json_data.get('width', 1)
    height = json_data.get('height', 1)
    ntiles = json_data.get('ntiles', 0)
    size = json_data.get('size', 0)  # DFA size
    tiles_data = json_data.get('tiles', [])
    
    # Parse tiles data - each tile has [q, s, fstart, fend, dstart] format
    tiles = []
    if tiles_data:
        for i in range(ntiles):
            base_idx = i * 5  # Each tile has 5 parameters
            if base_idx + 4 < len(tiles_data):
                tile = {
                    'q': tiles_data[base_idx],      # states
                    's': tiles_data[base_idx + 1],  # symbols  
                    'fstart': tiles_data[base_idx + 2],
                    'fend': tiles_data[base_idx + 3],
                    'dstart': tiles_data[base_idx + 4]  # DFA start position
                }
                tiles.append(tile)
    
    G = nx.Graph()
    
    # Board position nodes (type 0 - variables)
    # Weight by position difficulty: edges and corners are more constrained
    board_size = width * height
    for h in range(height):
        for w in range(width):
            pos_id = h * width + w
            
            # Calculate position constraint level
            edge_count = 0
            if h == 0 or h == height - 1:  # top/bottom edge
                edge_count += 1
            if w == 0 or w == width - 1:   # left/right edge  
                edge_count += 1
            
            # Corner positions are most constrained, edges less so, center least
            if edge_count == 2:  # corner
                constraint_level = 1.0
            elif edge_count == 1:  # edge
                constraint_level = 0.7
            else:  # interior
                constraint_level = 0.4
                
            G.add_node(f'pos_{h}_{w}', type=0, weight=constraint_level)
    
    # Tile placement constraint nodes (type 1)
    # Each tile creates complex placement constraints via DFA
    for t in range(ntiles):
        if t < len(tiles):
            tile = tiles[t]
            # Weight by DFA complexity (more states = more complex constraints)
            dfa_complexity = min(tile['q'] / 20.0, 1.0)  # normalize to [0,1]
            
            # Also consider symbol alphabet size
            symbol_complexity = min(tile['s'] / 10.0, 1.0)
            
            # Combined complexity score
            complexity = (dfa_complexity + symbol_complexity) / 2.0
            complexity = max(complexity, 0.3)  # minimum complexity
            
            G.add_node(f'tile_constraint_{t}', type=1, weight=complexity)
    
    # Boundary constraint nodes (type 1)
    # Right edge boundary constraint (forces ntiles+1 values)
    boundary_positions = height  # number of right-edge positions
    boundary_tightness = min(boundary_positions / board_size, 1.0)
    G.add_node('boundary_constraint', type=1, weight=boundary_tightness)
    
    # Non-boundary constraint (forbids ntiles+1 in other positions)  
    non_boundary_positions = board_size - boundary_positions
    non_boundary_tightness = min(non_boundary_positions / board_size, 1.0)
    G.add_node('non_boundary_constraint', type=1, weight=non_boundary_tightness)
    
    # Global ordering constraint (represents the regular constraint complexity)
    if size > 0:
        global_complexity = min(math.log(size) / 10.0, 1.0)  # log scale for large DFA
        G.add_node('global_regular_constraint', type=1, weight=global_complexity)
    
    # Edges: position-constraint participation (bipartite)
    for h in range(height):
        for w in range(width):
            pos = f'pos_{h}_{w}'
            pos_id = h * width + w
            
            # Connect to boundary constraints
            if w == width - 1:  # rightmost column
                G.add_edge(pos, 'boundary_constraint', weight=1.0)
            else:  # other positions
                G.add_edge(pos, 'non_boundary_constraint', weight=0.8)
            
            # Connect to global regular constraint with position-based weight
            if size > 0:
                # Central positions participate more strongly in global constraint
                center_h, center_w = height // 2, width // 2
                distance = abs(h - center_h) + abs(w - center_w)
                max_distance = center_h + center_w
                centrality = 1.0 - (distance / max(max_distance, 1))
                participation = 0.5 + 0.5 * centrality
                G.add_edge(pos, 'global_regular_constraint', weight=participation)
    
    # Connect positions to tile placement constraints
    # Each tile constraint affects all board positions, but with varying strength
    for t in range(ntiles):
        if t < len(tiles):
            tile = tiles[t]
            tile_constraint = f'tile_constraint_{t}'
            
            # Estimate tile size influence (more states usually means larger tiles)
            tile_influence = min(tile['q'] / 10.0, 1.0)
            
            for h in range(height):
                for w in range(width):
                    pos = f'pos_{h}_{w}'
                    
                    # Weight by tile influence and position importance
                    pos_weight = G.nodes[pos]['weight']
                    edge_weight = (tile_influence + pos_weight) / 2.0
                    edge_weight = max(edge_weight, 0.2)  # minimum participation
                    
                    G.add_edge(pos, tile_constraint, weight=edge_weight)
    
    # Add spatial conflict edges between adjacent positions
    # (for positions that might conflict in tile placement)
    conflict_threshold = 0.6  # only add conflicts for moderately complex instances
    if ntiles >= 3 and board_size >= 12:
        for h in range(height):
            for w in range(width - 1):  # horizontal adjacencies
                pos1 = f'pos_{h}_{w}'
                pos2 = f'pos_{h}_{w+1}'
                
                # Conflict strength based on position constraints
                weight1 = G.nodes[pos1]['weight']
                weight2 = G.nodes[pos2]['weight']
                conflict_strength = (weight1 + weight2) / 2.0
                
                if conflict_strength > conflict_threshold:
                    G.add_edge(pos1, pos2, weight=conflict_strength)
        
        for h in range(height - 1):  # vertical adjacencies
            for w in range(width):
                pos1 = f'pos_{h}_{w}'
                pos2 = f'pos_{h+1}_{w}'
                
                # Conflict strength based on position constraints
                weight1 = G.nodes[pos1]['weight']
                weight2 = G.nodes[pos2]['weight']
                conflict_strength = (weight1 + weight2) / 2.0
                
                if conflict_strength > conflict_threshold:
                    G.add_edge(pos1, pos2, weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()