#!/usr/bin/env python3
"""
Graph converter for pillars-planks-solution problem.
Created using subagent_prompt.md version: v_02

This problem is about 2D packing of planks and pillars where planks must be supported
by pillars at both ends, and pillars must sit on planks (or ground level).
Key challenges: structural dependencies, spatial constraints, height minimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the pillars-planks problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling structural dependencies
    - Plank nodes (type 0): decision variables for plank placement
    - Pillar nodes (type 0): decision variables for pillar placement  
    - Support constraints (type 1): planks need pillar support at both ends
    - Base constraints (type 1): pillars need to sit on planks or ground
    - Spatial constraints (type 1): non-overlapping requirements
    - Capacity constraints (type 1): fitting within available space
    
    Weight planks by relative width (larger planks harder to place)
    Weight pillars by height*width (more resource consumption)
    Weight constraints by tightness and scope
    """
    # Access data from json_data dict
    planks = json_data.get('planks', 0)
    pillars = json_data.get('pillars', 0)
    available_width = json_data.get('available_width', 1)
    available_height = json_data.get('available_height', 1)
    plank_width = json_data.get('plank_width', [])
    pillar_height = json_data.get('pillar_height', [])
    pillar_width = json_data.get('pillar_width', [])
    
    G = nx.Graph()
    
    # Calculate problem complexity metrics
    total_plank_width = sum(plank_width) if plank_width else 1
    total_pillar_volume = sum(pillar_height[i] * pillar_width[i] 
                             for i in range(min(len(pillar_height), len(pillar_width))))
    max_plank_width = max(plank_width) if plank_width else 1
    max_pillar_height = max(pillar_height) if pillar_height else 1
    max_pillar_width = max(pillar_width) if pillar_width else 1
    
    # Plank nodes (type 0) - weighted by relative size and placement difficulty
    for p in range(planks):
        if p < len(plank_width):
            width = plank_width[p]
            # Larger planks are harder to place (exponential scaling)
            size_difficulty = math.exp(3.0 * width / max_plank_width) - 1
            size_difficulty /= (math.exp(3.0) - 1)  # Normalize to [0,1]
            
            # Wide planks relative to available space are harder
            space_pressure = min(width / available_width, 1.0)
            
            # Combine factors
            weight = 0.3 + 0.7 * max(size_difficulty, space_pressure)
            G.add_node(f'plank_{p}', type=0, weight=weight)
    
    # Pillar nodes (type 0) - weighted by volume and structural importance
    for p in range(pillars):
        if p < len(pillar_height) and p < len(pillar_width):
            height = pillar_height[p]
            width = pillar_width[p]
            volume = height * width
            
            # Volume-based difficulty (logarithmic scaling)
            if total_pillar_volume > 0:
                volume_ratio = volume / (total_pillar_volume / pillars)
                volume_difficulty = math.log(1 + 2 * volume_ratio) / math.log(3)
            else:
                volume_difficulty = 0.5
                
            # Tall pillars create more structural complexity
            height_factor = height / max_pillar_height if max_pillar_height > 0 else 0.5
            
            # Wide pillars consume more horizontal space
            width_factor = width / max_pillar_width if max_pillar_width > 0 else 0.5
            
            weight = 0.2 + 0.3 * volume_difficulty + 0.3 * height_factor + 0.2 * width_factor
            G.add_node(f'pillar_{p}', type=0, weight=min(weight, 1.0))
    
    # Support constraints (type 1) - each plank needs support at both ends
    for p in range(planks):
        if p < len(plank_width):
            width = plank_width[p]
            # Wider planks have tighter support constraints
            support_tightness = min(width / available_width, 1.0)
            # Support constraints are critical - base weight 0.7
            weight = 0.7 + 0.3 * support_tightness
            
            # Left support constraint
            G.add_node(f'left_support_{p}', type=1, weight=weight)
            # Right support constraint  
            G.add_node(f'right_support_{p}', type=1, weight=weight)
    
    # Base/Foundation constraints (type 1) - each pillar sits on something
    for p in range(pillars):
        if p < len(pillar_height) and p < len(pillar_width):
            height = pillar_height[p]
            width = pillar_width[p]
            
            # Taller/wider pillars have tighter foundation requirements
            foundation_tightness = (height / max_pillar_height + width / max_pillar_width) / 2
            foundation_tightness = min(foundation_tightness, 1.0)
            
            weight = 0.6 + 0.4 * foundation_tightness
            G.add_node(f'foundation_{p}', type=1, weight=weight)
    
    # Spatial non-overlap constraints (type 1) - diffn constraint modeling
    # Create constraint nodes for spatial regions that are contested
    for p1 in range(planks):
        for p2 in range(p1 + 1, planks):
            if p1 < len(plank_width) and p2 < len(plank_width):
                # Spatial conflict potential based on size similarity
                w1, w2 = plank_width[p1], plank_width[p2]
                conflict_potential = 1.0 - abs(w1 - w2) / max(w1, w2)
                
                # Higher conflict for larger combined size
                size_factor = (w1 + w2) / (2 * available_width)
                weight = 0.4 + 0.6 * conflict_potential * min(size_factor, 1.0)
                
                G.add_node(f'spatial_conflict_plank_{p1}_{p2}', type=1, weight=weight)
    
    # Pillar-pillar spatial conflicts
    for p1 in range(pillars):
        for p2 in range(p1 + 1, pillars):
            if (p1 < len(pillar_height) and p1 < len(pillar_width) and 
                p2 < len(pillar_height) and p2 < len(pillar_width)):
                
                # Conflict based on volume and dimensional similarity
                vol1 = pillar_height[p1] * pillar_width[p1]
                vol2 = pillar_height[p2] * pillar_width[p2]
                
                volume_conflict = 1.0 - abs(vol1 - vol2) / max(vol1, vol2) if max(vol1, vol2) > 0 else 0.5
                
                # Size pressure factor
                total_volume = vol1 + vol2
                space_pressure = total_volume / (available_width * available_height / pillars)
                space_pressure = min(space_pressure, 1.0)
                
                weight = 0.3 + 0.4 * volume_conflict + 0.3 * space_pressure
                G.add_node(f'spatial_conflict_pillar_{p1}_{p2}', type=1, weight=weight)
    
    # Width capacity constraint (type 1)
    width_utilization = total_plank_width / available_width if available_width > 0 else 0
    width_pressure = min(width_utilization, 1.0)
    G.add_node('width_capacity', type=1, weight=0.5 + 0.5 * width_pressure)
    
    # Height capacity constraint (type 1) - estimated from pillar heights
    estimated_height_pressure = max_pillar_height / available_height if available_height > 0 else 0
    height_pressure = min(estimated_height_pressure, 1.0)
    G.add_node('height_capacity', type=1, weight=0.4 + 0.6 * height_pressure)
    
    # Bipartite edges: variable-constraint participation
    
    # Plank participation in support constraints
    for p in range(planks):
        plank_node = f'plank_{p}'
        if G.has_node(plank_node):
            # Each plank participates in its own support constraints
            if G.has_node(f'left_support_{p}'):
                G.add_edge(plank_node, f'left_support_{p}', weight=1.0)
            if G.has_node(f'right_support_{p}'):
                G.add_edge(plank_node, f'right_support_{p}', weight=1.0)
            
            # Plank participates in width capacity
            if G.has_node('width_capacity') and p < len(plank_width):
                contribution = plank_width[p] / total_plank_width if total_plank_width > 0 else 0.5
                G.add_edge(plank_node, 'width_capacity', weight=contribution)
    
    # Pillar participation in foundation and support constraints
    for p in range(pillars):
        pillar_node = f'pillar_{p}'
        if G.has_node(pillar_node):
            # Each pillar participates in its foundation constraint
            if G.has_node(f'foundation_{p}'):
                G.add_edge(pillar_node, f'foundation_{p}', weight=1.0)
            
            # Pillars can potentially support any plank (structural coupling)
            for plank_p in range(planks):
                if G.has_node(f'left_support_{plank_p}'):
                    # Weight by geometric compatibility
                    if (p < len(pillar_height) and plank_p < len(plank_width) and 
                        p < len(pillar_width)):
                        # Structural stability factor
                        stability = min(pillar_height[p] / max_pillar_height, 1.0) if max_pillar_height > 0 else 0.5
                        G.add_edge(pillar_node, f'left_support_{plank_p}', weight=0.3 + 0.7 * stability)
                        G.add_edge(pillar_node, f'right_support_{plank_p}', weight=0.3 + 0.7 * stability)
            
            # Pillar participates in height capacity
            if G.has_node('height_capacity') and p < len(pillar_height):
                height_contribution = pillar_height[p] / max_pillar_height if max_pillar_height > 0 else 0.5
                G.add_edge(pillar_node, 'height_capacity', weight=height_contribution)
    
    # Spatial conflict edges
    for p1 in range(planks):
        for p2 in range(p1 + 1, planks):
            conflict_node = f'spatial_conflict_plank_{p1}_{p2}'
            if G.has_node(conflict_node):
                if G.has_node(f'plank_{p1}'):
                    G.add_edge(f'plank_{p1}', conflict_node, weight=0.8)
                if G.has_node(f'plank_{p2}'):
                    G.add_edge(f'plank_{p2}', conflict_node, weight=0.8)
    
    for p1 in range(pillars):
        for p2 in range(p1 + 1, pillars):
            conflict_node = f'spatial_conflict_pillar_{p1}_{p2}'
            if G.has_node(conflict_node):
                if G.has_node(f'pillar_{p1}'):
                    G.add_edge(f'pillar_{p1}', conflict_node, weight=0.8)
                if G.has_node(f'pillar_{p2}'):
                    G.add_edge(f'pillar_{p2}', conflict_node, weight=0.8)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()