#!/usr/bin/env python3
"""
Graph converter for product-and-shelves problem.
Created using subagent_prompt.md version: v_02

This problem is about 3D bin packing with products (3D boxes) that need to be placed 
on shelves (3D bins) to minimize the number of shelves used.
Key challenges: 3D space constraints, non-overlapping placement, shelf capacity limits
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the product-and-shelves problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with products as variables and constraints for:
    - Shelf capacity constraints (one per shelf per dimension)
    - Non-overlapping constraints (one per shelf)
    - Overall bin minimization pressure
    
    Key entities:
    - Items (instances of products) as Type 0 nodes
    - Shelf capacity constraints as Type 1 nodes  
    - Shelf non-overlap constraints as Type 1 nodes
    - Shelves as Type 2 resource nodes
    
    Weights based on:
    - Item volume/complexity for variable weights
    - Constraint tightness for constraint weights
    - Shelf utilization pressure for resource weights
    """
    
    # Extract data
    nr_shelves = json_data.get('nr_shelves', 1)
    shelves = json_data.get('shelves', [1, 1, 1])  # [length, width, height]
    nr_products = json_data.get('nr_products', [])
    product_size = json_data.get('product_size', [])
    
    # Calculate derived values
    total_items = sum(nr_products) if nr_products else 0
    shelf_volume = shelves[0] * shelves[1] * shelves[2] if len(shelves) >= 3 else 1
    total_shelf_capacity = nr_shelves * shelf_volume
    
    # Reconstruct product sizes as 3D (length, width, height)
    # product_size is flattened: [p1_l, p1_w, p1_h, p2_l, p2_w, p2_h, ...]
    product_3d_sizes = []
    for i in range(0, len(product_size), 3):
        if i + 2 < len(product_size):
            product_3d_sizes.append([
                product_size[i],     # length
                product_size[i+1],   # width  
                product_size[i+2]    # height
            ])
    
    # Calculate total volume demand
    total_product_volume = 0
    for i, count in enumerate(nr_products):
        if i < len(product_3d_sizes):
            product_vol = product_3d_sizes[i][0] * product_3d_sizes[i][1] * product_3d_sizes[i][2]
            total_product_volume += count * product_vol
    
    G = nx.Graph()
    
    # Type 0: Item nodes (instances of products)
    item_id = 0
    max_item_volume = 1
    
    # Calculate max volume for normalization
    for i, count in enumerate(nr_products):
        if i < len(product_3d_sizes):
            vol = product_3d_sizes[i][0] * product_3d_sizes[i][1] * product_3d_sizes[i][2]
            max_item_volume = max(max_item_volume, vol)
    
    # Add item nodes
    for product_type, count in enumerate(nr_products):
        if product_type < len(product_3d_sizes):
            dims = product_3d_sizes[product_type]
            volume = dims[0] * dims[1] * dims[2]
            
            # Weight by relative volume and aspect ratio complexity
            aspect_ratio = max(dims) / max(min(dims), 1)
            complexity = (volume / max_item_volume) * (1.0 + math.log(aspect_ratio) / 3.0)
            weight = min(complexity, 1.0)
            
            for instance in range(count):
                G.add_node(f'item_{item_id}', type=0, weight=weight)
                item_id += 1
    
    # Type 1: Constraint nodes
    
    # Shelf capacity constraints (one per shelf per dimension)
    for shelf_id in range(nr_shelves):
        for dim in range(3):  # length, width, height
            dim_names = ['length', 'width', 'height']
            shelf_capacity = shelves[dim] if dim < len(shelves) else 1
            
            # Calculate potential demand for this dimension
            total_dim_demand = sum(
                nr_products[i] * product_3d_sizes[i][dim] 
                for i in range(min(len(nr_products), len(product_3d_sizes)))
            )
            
            # Tightness based on demand vs capacity
            if total_dim_demand > 0:
                tightness = min(total_dim_demand / (shelf_capacity * nr_shelves), 1.0)
            else:
                tightness = 0.5
                
            G.add_node(f'shelf_{shelf_id}_{dim_names[dim]}_capacity', 
                      type=1, weight=tightness)
    
    # Non-overlapping constraints (one per shelf)
    for shelf_id in range(nr_shelves):
        # Weight by expected congestion
        avg_items_per_shelf = total_items / nr_shelves if nr_shelves > 0 else 1
        congestion = min(avg_items_per_shelf / 10.0, 1.0)  # Normalize assuming 10+ items is high
        G.add_node(f'shelf_{shelf_id}_nonoverlap', type=1, weight=congestion)
    
    # Global bin minimization pressure constraint
    volume_pressure = min(total_product_volume / total_shelf_capacity, 1.0) if total_shelf_capacity > 0 else 0.5
    G.add_node('minimize_shelves', type=1, weight=volume_pressure)
    
    # Type 2: Shelf resource nodes
    for shelf_id in range(nr_shelves):
        # Weight by scarcity - later shelves are more "expensive"
        scarcity = 1.0 - (shelf_id / max(nr_shelves - 1, 1))
        G.add_node(f'shelf_{shelf_id}', type=2, weight=scarcity)
    
    # Edges: Connect items to constraints they participate in
    
    item_id = 0
    for product_type, count in enumerate(nr_products):
        if product_type < len(product_3d_sizes):
            dims = product_3d_sizes[product_type]
            
            for instance in range(count):
                item_node = f'item_{item_id}'
                
                # Connect to all shelf capacity constraints
                for shelf_id in range(nr_shelves):
                    for dim in range(3):
                        dim_names = ['length', 'width', 'height']
                        constraint_node = f'shelf_{shelf_id}_{dim_names[dim]}_capacity'
                        
                        # Edge weight by consumption ratio
                        shelf_capacity = shelves[dim] if dim < len(shelves) else 1
                        consumption_ratio = dims[dim] / shelf_capacity if shelf_capacity > 0 else 0.5
                        edge_weight = min(consumption_ratio, 1.0)
                        
                        G.add_edge(item_node, constraint_node, weight=edge_weight)
                
                # Connect to all non-overlap constraints
                for shelf_id in range(nr_shelves):
                    constraint_node = f'shelf_{shelf_id}_nonoverlap'
                    # Weight by item volume relative to shelf volume
                    item_volume = dims[0] * dims[1] * dims[2]
                    volume_ratio = item_volume / shelf_volume if shelf_volume > 0 else 0.5
                    edge_weight = min(volume_ratio, 1.0)
                    
                    G.add_edge(item_node, constraint_node, weight=edge_weight)
                
                # Connect to minimize_shelves constraint
                # Weight by item's contribution to total volume
                item_volume = dims[0] * dims[1] * dims[2]
                volume_contribution = item_volume / total_product_volume if total_product_volume > 0 else (1.0 / total_items)
                G.add_edge(item_node, 'minimize_shelves', weight=volume_contribution)
                
                # Connect to shelf resources (potential placement)
                for shelf_id in range(nr_shelves):
                    shelf_node = f'shelf_{shelf_id}'
                    # Weight by fit quality - exponential decay for tight fits
                    fit_ratios = [dims[d] / shelves[d] if d < len(shelves) and shelves[d] > 0 else 1.0 for d in range(3)]
                    max_fit_ratio = max(fit_ratios)
                    
                    if max_fit_ratio <= 1.0:  # Item can fit
                        # Better fit = higher weight
                        fit_weight = math.exp(-2.0 * max_fit_ratio)
                    else:  # Item cannot fit
                        fit_weight = 0.0
                    
                    G.add_edge(item_node, shelf_node, weight=fit_weight)
                
                item_id += 1
    
    # Add conflict edges between large items competing for limited space
    if total_product_volume > 0.8 * total_shelf_capacity:  # Tight packing scenario
        large_items = []
        item_id = 0
        
        for product_type, count in enumerate(nr_products):
            if product_type < len(product_3d_sizes):
                dims = product_3d_sizes[product_type]
                volume = dims[0] * dims[1] * dims[2]
                
                # Consider items above average volume as "large"
                avg_volume = total_product_volume / total_items if total_items > 0 else 0
                if volume > avg_volume:
                    for instance in range(count):
                        large_items.append((f'item_{item_id}', volume))
                        item_id += 1
                else:
                    item_id += count
        
        # Add conflict edges between large items
        for i in range(len(large_items)):
            for j in range(i+1, min(len(large_items), i+6)):  # Limit conflicts to avoid dense graph
                item1, vol1 = large_items[i]
                item2, vol2 = large_items[j]
                
                # Conflict strength based on combined volume pressure
                combined_volume = vol1 + vol2
                conflict_strength = min(combined_volume / shelf_volume, 1.0) if shelf_volume > 0 else 0.5
                
                G.add_edge(item1, item2, weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()