#!/usr/bin/env python3
"""
Graph converter for prop_stress problem.
Created using subagent_prompt.md version: v_02

This problem is about propagation stress - testing constraint propagation with chains of inequalities.
Key challenges: Complex propagation chains, unsatisfiable instances designed to stress solvers,
multiple interacting constraint types creating cascading propagation effects.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the propagation stress problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the propagation chains and variable interactions
    - Variables: y[0..n] and x[0..m] arrays as decision points
    - Constraints: Chain constraints, coupling constraints, and final contradiction
    - Weight by propagation strength and constraint tightness
    - Model the inherent unsatisfiability stress patterns
    """
    # Access data directly from json_data dict
    k = json_data.get('k', 1)
    n = json_data.get('n', 1) 
    m = json_data.get('m', 1)
    
    # Create graph
    G = nx.Graph()
    
    # Variable nodes for y array [0..n]
    # Weight by position in chain - early variables more constrained
    for i in range(n + 1):
        # Higher weight for variables that appear in more constraints
        constraint_count = 0
        if i == 0:  # y[0] appears in coupling and final constraints
            constraint_count = n + 1  
        elif i == n:  # y[n] appears in coupling constraint
            constraint_count = 1
        else:  # y[i] appears in chain constraints
            constraint_count = 2 if i > 1 else 1
            
        # Normalize by maximum possible constraints
        weight = min(constraint_count / (n + 2), 1.0)
        G.add_node(f'y_{i}', type=0, weight=weight)
    
    # Variable nodes for x array [0..m]
    # Weight by position in ordering chain and coupling involvement
    for i in range(m + 1):
        if i == 0:  # x[0] coupled with y[n]
            weight = 0.8
        elif i == m:  # x[m] creates final contradiction with y[0]
            weight = 1.0  # Highest weight - creates unsatisfiability
        else:  # x[i] in ordering chain
            # Variables in middle of chain are less constrained
            chain_position = abs(i - m/2) / (m/2)  # Distance from center
            weight = 0.3 + 0.4 * chain_position
            
        G.add_node(f'x_{i}', type=0, weight=weight)
    
    # Constraint nodes with tightness-based weights
    
    # Chain constraints for y array: y[i-1] - y[i] <= 0 (i=2..n)
    for i in range(2, n + 1):
        # These enforce ordering, weight by chain position
        chain_pos = i / n
        weight = 0.6 + 0.3 * chain_pos  # Later in chain = tighter
        G.add_node(f'y_chain_{i}', type=1, weight=weight)
        
        # Connect variables to constraint
        G.add_edge(f'y_{i-1}', f'y_chain_{i}', weight=0.8)
        G.add_edge(f'y_{i}', f'y_chain_{i}', weight=0.8)
    
    # Coupling constraints: y[0] - y[i] <= n - i + 1 (i=1..n)
    for i in range(1, n + 1):
        # Tightness increases as bound decreases (larger i)
        bound = n - i + 1
        tightness = 1.0 - (bound / n) if n > 0 else 0.5
        weight = 0.5 + 0.4 * tightness
        G.add_node(f'y_coupling_{i}', type=1, weight=weight)
        
        # Connect variables to constraint
        G.add_edge(f'y_0', f'y_coupling_{i}', weight=0.7)
        G.add_edge(f'y_{i}', f'y_coupling_{i}', weight=0.7)
    
    # Coupling constraint: y[n] - x[0] <= 0
    G.add_node('yx_coupling', type=1, weight=0.8)
    G.add_edge(f'y_{n}', 'yx_coupling', weight=0.9)
    G.add_edge('x_0', 'yx_coupling', weight=0.9)
    
    # Ordering constraints for x array: x[i] - x[j] <= 0 (i=0..m-1, j=i+1..m)
    constraint_id = 0
    for i in range(m):
        for j in range(i + 1, m + 1):
            # Weight by span of ordering constraint
            span = j - i
            max_span = m
            # Longer spans are easier to satisfy
            weight = 0.4 + 0.4 * (1.0 - span / max_span) if max_span > 0 else 0.6
            
            G.add_node(f'x_order_{constraint_id}', type=1, weight=weight)
            
            # Connect variables with weights based on constraint involvement
            edge_weight = 0.6 + 0.3 * (span / max_span) if max_span > 0 else 0.8
            G.add_edge(f'x_{i}', f'x_order_{constraint_id}', weight=edge_weight)
            G.add_edge(f'x_{j}', f'x_order_{constraint_id}', weight=edge_weight)
            
            constraint_id += 1
    
    # Final contradiction constraint: x[m] - y[0] <= -2
    # This creates the unsatisfiability - highest weight
    G.add_node('contradiction', type=1, weight=1.0)
    G.add_edge(f'x_{m}', 'contradiction', weight=1.0)
    G.add_edge('y_0', 'contradiction', weight=1.0)
    
    # Add resource node representing the propagation stress
    # Weight by problem scale (k*n domain size creates stress)
    domain_size = k * n
    stress_level = min(math.log10(domain_size + 1) / 4.0, 1.0)  # Logarithmic scaling
    G.add_node('propagation_stress', type=2, weight=stress_level)
    
    # Connect stress node to key variables that create propagation pressure
    G.add_edge('propagation_stress', 'y_0', weight=0.9)  # Central hub
    G.add_edge('propagation_stress', f'x_{m}', weight=0.9)  # Contradiction source
    G.add_edge('propagation_stress', f'y_{n}', weight=0.7)  # Coupling point
    G.add_edge('propagation_stress', 'x_0', weight=0.7)    # Coupling point
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()