#!/usr/bin/env python3
"""
Graph converter for quasigroup7 problem.
Created using subagent_prompt.md version: v_02

This problem is about constructing an n×n quasigroup (multiplication table) where:
- Each row and column contains all values 0 to n-1 exactly once (Latin square)
- Diagonal elements equal their index (quasiGroup[i,i] = i) 
- Axiom 7 constraint: (b*a)*b = a*(b*a) must hold
- Additional constraint: quasiGroup[i,n-1] + 2 >= i

Key challenges: Complex interdependencies between cells due to axiom 7, 
diagonal constraints, and Latin square properties create highly constrained structure.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the quasigroup7 problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the complex constraint structure
    - Variable nodes (type 0): Each cell in the n×n quasigroup table
    - Constraint nodes (type 1): Row/column all-different, diagonal, axiom 7, and implied constraints
    - Weights reflect constraint complexity and variable criticality
    
    The axiom 7 constraint creates complex interdependencies that make this problem particularly challenging.
    Diagonal constraints and the implied constraint add additional structural complexity.
    """
    n = json_data.get('n', 5)
    
    G = nx.Graph()
    
    # Variable nodes: each cell in the n×n quasigroup table
    for i in range(n):
        for j in range(n):
            # Weight by position criticality - diagonal cells are fixed, corner/edge cells more constrained
            if i == j:
                # Diagonal cells are fixed (constrained to value i)
                weight = 1.0
            else:
                # Non-diagonal cells weighted by their constraint involvement
                # Corner and edge cells participate in fewer axiom 7 constraints
                edge_penalty = 0.1 if (i == 0 or i == n-1 or j == 0 or j == n-1) else 0.0
                weight = 0.7 + edge_penalty + 0.2 * (abs(i - j) / n)  # Distance from diagonal adds complexity
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=weight)
    
    # Constraint nodes for different constraint types
    
    # 1. Row all-different constraints (n constraints)
    for i in range(n):
        # Weight by row involvement in axiom 7 - rows involved in more axiom 7 constraints are tighter
        axiom7_involvement = sum(1 for j in range(n) for k in range(n) if True)  # All positions involved
        weight = 0.8 + 0.2 * (i / n)  # Slight variation by row position
        G.add_node(f'row_alldiff_{i}', type=1, weight=weight)
    
    # 2. Column all-different constraints (n constraints)  
    for j in range(n):
        weight = 0.8 + 0.2 * (j / n)  # Slight variation by column position
        G.add_node(f'col_alldiff_{j}', type=1, weight=weight)
    
    # 3. Diagonal constraints (n constraints) - these are equality constraints
    for i in range(n):
        weight = 1.0  # These are tight equality constraints
        G.add_node(f'diagonal_{i}', type=1, weight=weight)
    
    # 4. Axiom 7 constraints: quasiGroup[i, quasiGroup[j,i]] = quasiGroup[quasiGroup[j,i],j]
    # One constraint for each (i,j) pair
    for i in range(n):
        for j in range(n):
            # Weight by complexity - constraints involving more variable interactions are harder
            # This constraint involves 4 cells: (i,quasiGroup[j,i]), (quasiGroup[j,i],j), (j,i), and indirectly others
            complexity = 1.0 - 0.3 * math.exp(-2.0 * abs(i - j) / n)  # Non-linear complexity based on distance
            G.add_node(f'axiom7_{i}_{j}', type=1, weight=complexity)
    
    # 5. Implied constraints: quasiGroup[i,n-1] + 2 >= i (n constraints)
    for i in range(n):
        # These are inequality constraints, generally less tight than equalities
        tightness = 0.6 + 0.3 * (i / n)  # Later rows more constrained
        G.add_node(f'implied_{i}', type=1, weight=tightness)
    
    # Edges: Variable-Constraint participation (bipartite structure)
    
    # Row all-different constraint edges
    for i in range(n):
        for j in range(n):
            G.add_edge(f'cell_{i}_{j}', f'row_alldiff_{i}', weight=1.0)
    
    # Column all-different constraint edges
    for i in range(n):
        for j in range(n):
            G.add_edge(f'cell_{i}_{j}', f'col_alldiff_{j}', weight=1.0)
    
    # Diagonal constraint edges
    for i in range(n):
        G.add_edge(f'cell_{i}_{i}', f'diagonal_{i}', weight=1.0)
    
    # Axiom 7 constraint edges - each constraint involves multiple cells
    for i in range(n):
        for j in range(n):
            constraint_id = f'axiom7_{i}_{j}'
            # Primary cells directly involved
            G.add_edge(f'cell_{j}_{i}', constraint_id, weight=1.0)  # quasiGroup[j,i]
            # Secondary involvement - all cells in row i and column j could be the result
            for k in range(n):
                participation_weight = 0.6 + 0.4 * math.exp(-abs(k - i) / n)  # Non-linear participation
                G.add_edge(f'cell_{i}_{k}', constraint_id, weight=participation_weight)  # quasiGroup[i,*]
                G.add_edge(f'cell_{k}_{j}', constraint_id, weight=participation_weight)  # quasiGroup[*,j]
    
    # Implied constraint edges
    for i in range(n):
        G.add_edge(f'cell_{i}_{n-1}', f'implied_{i}', weight=1.0)
    
    # Add conflict edges for highly constrained variable pairs
    # Variables that share many constraints and are likely to conflict
    for i1 in range(n):
        for j1 in range(n):
            for i2 in range(n):
                for j2 in range(n):
                    if (i1, j1) >= (i2, j2):  # Avoid duplicate edges
                        continue
                    
                    conflicts = 0
                    # Same row conflict
                    if i1 == i2:
                        conflicts += 1
                    # Same column conflict  
                    if j1 == j2:
                        conflicts += 1
                    # Axiom 7 interdependency
                    if (i1 == j2 and j1 == i2) or (i1 == i2 and j1 == j2):
                        conflicts += 2
                    
                    if conflicts >= 2:  # High conflict potential
                        conflict_weight = min(0.8, conflicts * 0.3)
                        G.add_edge(f'cell_{i1}_{j1}', f'cell_{i2}_{j2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()