#!/usr/bin/env python3
"""
Graph converter for N-Queens problem.
Created using subagent_prompt.md version: v_02

This problem is about placing n queens on an n×n chessboard such that no two queens attack each other.
Key challenges: Queens attack along rows, columns, and diagonals, creating complex constraint interactions.
The difficulty grows exponentially with board size due to increasing constraint density and search space.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the N-Queens problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Pure bipartite model with explicit constraint nodes
    - Type 0 nodes: Board positions (variables in the CSP)
    - Type 1 nodes: Constraints (row, column, diagonal constraints)
    - Edges: Variable participation in constraints
    - Weights: Position centrality and constraint scope-based weighting
    
    The graph captures the constraint structure that makes N-Queens challenging:
    - Central positions are more constrained (higher weight)
    - Diagonal constraints have variable scope (fewer positions on edges)
    - All constraints are equally critical (no queen attacks allowed)
    """
    n = json_data.get('n', 8)
    
    G = nx.Graph()
    
    # Type 0 nodes: Board positions (decision variables)
    # Weight by constraint density - central positions are more constrained
    for r in range(n):
        for c in range(n):
            # Calculate centrality weight: positions closer to center have more constraints
            # Use non-linear weighting to emphasize the difference
            distance_from_center = abs(r - (n-1)/2) + abs(c - (n-1)/2)
            max_distance = (n-1)  # Distance from center to corner
            centrality = 1.0 - (distance_from_center / max_distance)
            # Apply exponential to emphasize central positions more
            weight = 0.3 + 0.7 * math.exp(2 * centrality - 2)  # Range [0.3, 1.0]
            G.add_node(f'pos_{r}_{c}', type=0, weight=weight)
    
    # Type 1 nodes: Constraint nodes with scope-based weighting
    
    # Row constraints (each has scope = n)
    row_weight = 1.0  # All rows have same scope
    for r in range(n):
        G.add_node(f'row_{r}', type=1, weight=row_weight)
    
    # Column constraints (each has scope = n)  
    col_weight = 1.0  # All columns have same scope
    for c in range(n):
        G.add_node(f'col_{c}', type=1, weight=col_weight)
    
    # Main diagonal constraints (variable scope)
    # Diagonals going from top-left to bottom-right
    for d in range(2*n-1):
        # Calculate scope (number of positions on this diagonal)
        if d < n:
            scope = d + 1
        else:
            scope = 2*n - 1 - d
        # Weight by scope normalized to [0.4, 1.0]
        weight = 0.4 + 0.6 * (scope / n)
        G.add_node(f'diag_main_{d}', type=1, weight=weight)
    
    # Anti-diagonal constraints (variable scope)
    # Diagonals going from top-right to bottom-left
    for d in range(2*n-1):
        if d < n:
            scope = d + 1
        else:
            scope = 2*n - 1 - d
        weight = 0.4 + 0.6 * (scope / n)
        G.add_node(f'diag_anti_{d}', type=1, weight=weight)
    
    # Bipartite edges: Connect each position to all constraints it participates in
    for r in range(n):
        for c in range(n):
            pos = f'pos_{r}_{c}'
            
            # Connect to row constraint
            G.add_edge(pos, f'row_{r}', weight=1.0)
            
            # Connect to column constraint
            G.add_edge(pos, f'col_{c}', weight=1.0)
            
            # Connect to main diagonal constraint
            # Main diagonal: positions where (r - c) is constant
            main_diag_idx = r - c + (n - 1)  # Offset to make non-negative
            G.add_edge(pos, f'diag_main_{main_diag_idx}', weight=1.0)
            
            # Connect to anti-diagonal constraint
            # Anti-diagonal: positions where (r + c) is constant
            anti_diag_idx = r + c
            G.add_edge(pos, f'diag_anti_{anti_diag_idx}', weight=1.0)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()