#!/usr/bin/env python3
"""
Graph converter for Resource Availability Cost Problem (RACP).
Created using subagent_prompt.md version: v_02

This problem is about scheduling tasks with precedence constraints while minimizing
the cost of acquiring sufficient resource capacity to meet all resource demands.
Key challenges: Resource cost optimization, temporal scheduling, precedence constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the RACP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a rich bipartite graph modeling the resource-task interaction structure.
    - Task nodes (type 0): Weight by resource intensity and duration
    - Resource constraint nodes (type 1): Weight by cost and scarcity
    - Precedence constraint nodes (type 1): Weight by criticality in dependency chain
    - Edges reflect resource consumption and precedence relationships
    """
    # Extract problem data
    n_tasks = json_data.get('n_tasks', 0)
    n_res = json_data.get('n_res', 0)
    t_max = json_data.get('t_max', 1)
    
    dur = json_data.get('dur', [])
    cost = json_data.get('cost', [])
    succ = json_data.get('succ', [])
    
    # Reconstruct 2D resource requirements matrix from flattened array
    rr_flat = json_data.get('rr', [])
    rr = []
    for r in range(n_res):
        rr.append(rr_flat[r * n_tasks:(r + 1) * n_tasks] if len(rr_flat) >= (r + 1) * n_tasks else [0] * n_tasks)
    
    G = nx.Graph()
    
    # Calculate derived metrics for better weights
    max_duration = max(dur) if dur else 1
    max_cost = max(cost) if cost else 1
    total_resource_demand = [sum(rr[r]) for r in range(n_res)] if rr else [1] * n_res
    max_total_demand = max(total_resource_demand) if total_resource_demand else 1
    
    # Task nodes (type 0) - weighted by resource intensity and temporal criticality
    for i in range(n_tasks):
        task_duration = dur[i] if i < len(dur) else 1
        
        # Calculate total resource consumption across all resources
        total_resource_usage = sum(rr[r][i] for r in range(n_res) if i < len(rr[r]))
        
        # Calculate weighted resource consumption (by resource cost)
        weighted_consumption = sum(cost[r] * rr[r][i] for r in range(min(n_res, len(cost))) if i < len(rr[r]))
        max_weighted_consumption = sum(cost[r] * max(rr[r]) for r in range(min(n_res, len(cost))) if rr[r])
        max_weighted_consumption = max(max_weighted_consumption, 1)
        
        # Weight combines resource intensity and temporal impact
        duration_factor = task_duration / max_duration
        resource_factor = weighted_consumption / max_weighted_consumption
        
        # Non-linear combination emphasizing high-impact tasks
        task_weight = math.sqrt(duration_factor * resource_factor)
        
        G.add_node(f'task_{i}', type=0, weight=min(task_weight, 1.0))
    
    # Resource constraint nodes (type 1) - one per resource, weighted by cost and demand
    for r in range(n_res):
        resource_cost = cost[r] if r < len(cost) else 1
        resource_demand = total_resource_demand[r] if r < len(total_resource_demand) else 1
        
        # Weight by normalized cost and relative scarcity
        cost_factor = resource_cost / max_cost
        demand_factor = resource_demand / max_total_demand
        
        # Higher weight for expensive, high-demand resources
        resource_weight = (cost_factor + demand_factor) / 2.0
        
        G.add_node(f'resource_{r}', type=1, weight=min(resource_weight, 1.0))
    
    # Precedence constraint nodes (type 1) - one per precedence relationship
    precedence_id = 0
    task_predecessors = [[] for _ in range(n_tasks)]
    
    # Build predecessor lists
    for i in range(n_tasks):
        if i < len(succ):
            for j in succ[i]:
                if isinstance(j, int) and 1 <= j <= n_tasks:
                    task_predecessors[j - 1].append(i)  # Convert to 0-based indexing
    
    # Create precedence constraint nodes
    for i in range(n_tasks):
        for j_1based in (succ[i] if i < len(succ) else []):
            if isinstance(j_1based, int) and 1 <= j_1based <= n_tasks:
                j = j_1based - 1  # Convert to 0-based
                
                # Weight precedence by the criticality of the dependency
                pred_duration = dur[i] if i < len(dur) else 1
                succ_duration = dur[j] if j < len(dur) else 1
                
                # Longer dependencies are more critical
                criticality = (pred_duration + succ_duration) / (2 * max_duration)
                
                G.add_node(f'prec_{precedence_id}', type=1, weight=min(criticality, 1.0))
                precedence_id += 1
    
    # Task-Resource edges - bipartite connections weighted by consumption intensity
    for i in range(n_tasks):
        for r in range(n_res):
            if r < len(rr) and i < len(rr[r]) and rr[r][i] > 0:
                # Resource consumption relative to maximum demand for this resource
                max_demand_for_resource = max(rr[r]) if rr[r] else 1
                consumption_ratio = rr[r][i] / max_demand_for_resource
                
                # Apply exponential weighting to emphasize high consumers
                edge_weight = min(1.0 - math.exp(-3.0 * consumption_ratio), 1.0)
                
                G.add_edge(f'task_{i}', f'resource_{r}', weight=edge_weight)
    
    # Task-Precedence edges - connect tasks to their precedence constraints
    precedence_id = 0
    for i in range(n_tasks):
        for j_1based in (succ[i] if i < len(succ) else []):
            if isinstance(j_1based, int) and 1 <= j_1based <= n_tasks:
                j = j_1based - 1  # Convert to 0-based
                
                # Both predecessor and successor connect to the precedence constraint
                G.add_edge(f'task_{i}', f'prec_{precedence_id}', weight=1.0)
                G.add_edge(f'task_{j}', f'prec_{precedence_id}', weight=1.0)
                
                precedence_id += 1
    
    # Add conflict edges between tasks competing for over-subscribed resources
    for r in range(n_res):
        if r < len(rr):
            # Find tasks that use this resource significantly
            high_consumers = [(i, rr[r][i]) for i in range(n_tasks) if i < len(rr[r]) and rr[r][i] > 0]
            high_consumers.sort(key=lambda x: x[1], reverse=True)
            
            # If many tasks compete for this resource, add conflict edges
            if len(high_consumers) > 3:
                resource_capacity_estimate = max(rr[r]) * 2  # Rough estimate
                
                for idx1 in range(min(len(high_consumers), 6)):
                    for idx2 in range(idx1 + 1, min(len(high_consumers), 6)):
                        i1, consumption1 = high_consumers[idx1]
                        i2, consumption2 = high_consumers[idx2]
                        
                        # Add conflict edge if combined consumption suggests potential conflict
                        if consumption1 + consumption2 > resource_capacity_estimate * 0.7:
                            conflict_strength = (consumption1 + consumption2) / (resource_capacity_estimate * 2)
                            G.add_edge(f'task_{i1}', f'task_{i2}', weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()