#!/usr/bin/env python3
"""
Graph converter for Radiation problem.
Created using subagent_prompt.md version: v_02

This problem is about radiation therapy planning where we need to deliver specific 
radiation intensities to cells in a grid using shape matrices and beam-on times.
Key challenges: minimizing beam time while satisfying intensity requirements,
dealing with complex 3D constraint relationships between cells and beam configurations.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the radiation therapy planning problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with cells as variables and constraints
    - Cell nodes (type 0): Each grid cell needing radiation
    - Intensity constraint nodes (type 1): Constraints ensuring proper intensity delivery
    - Row constraint nodes (type 1): Upper bound constraints for each row and beam time
    - Beam time constraints (type 1): Beam time and shape matrix count constraints
    - Edge weights reflect intensity requirements and constraint tightness
    """
    m = json_data.get('m', 0)  # rows
    n = json_data.get('n', 0)  # columns
    intensity_flat = json_data.get('Intensity', [])
    
    # Reshape flat intensity array to 2D matrix
    intensity = []
    for i in range(m):
        row = []
        for j in range(n):
            idx = i * n + j
            if idx < len(intensity_flat):
                row.append(intensity_flat[idx])
            else:
                row.append(0)
        intensity.append(row)
    
    G = nx.Graph()
    
    # Calculate problem statistics for weight normalization
    max_intensity = max(max(row) for row in intensity) if intensity else 1
    total_intensity = sum(sum(row) for row in intensity)
    avg_intensity = total_intensity / (m * n) if m * n > 0 else 0
    
    # Add cell nodes (type 0) - decision variables for radiation exposure
    for i in range(m):
        for j in range(n):
            cell_intensity = intensity[i][j]
            
            # Weight by intensity requirement (higher intensity = more critical)
            if max_intensity > 0:
                intensity_weight = cell_intensity / max_intensity
            else:
                intensity_weight = 0.5
                
            # Add spatial complexity - corner/edge cells are harder to treat
            edge_penalty = 0
            if i == 0 or i == m-1:  # top/bottom edge
                edge_penalty += 0.1
            if j == 0 or j == n-1:  # left/right edge
                edge_penalty += 0.1
                
            # Cells with zero intensity are easier (don't need treatment)
            if cell_intensity == 0:
                complexity_weight = 0.1
            else:
                # Non-linear scaling for high intensity cells
                complexity_weight = min(0.3 + 0.6 * (intensity_weight ** 0.7) + edge_penalty, 1.0)
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=complexity_weight)
    
    # Add intensity constraint nodes (type 1) - one per cell with non-zero intensity
    for i in range(m):
        for j in range(n):
            if intensity[i][j] > 0:
                # Weight by constraint tightness (higher intensity = tighter constraint)
                intensity_weight = intensity[i][j] / max_intensity if max_intensity > 0 else 0.5
                
                # High intensity requirements create tighter constraints
                constraint_tightness = 0.4 + 0.6 * (intensity_weight ** 0.8)
                
                G.add_node(f'intensity_constraint_{i}_{j}', type=1, weight=constraint_tightness)
                
                # Connect cell to its intensity constraint
                G.add_edge(f'cell_{i}_{j}', f'intensity_constraint_{i}_{j}', 
                          weight=min(intensity_weight * 1.2, 1.0))
    
    # Add row constraint nodes (type 1) for upper_bound_on_increments constraints
    # These are complex constraints that limit shape matrix configurations
    max_beam_time = max_intensity if max_intensity > 0 else 1
    
    for i in range(m):
        for b in range(1, max_beam_time + 1):
            # Weight by beam time and row complexity
            beam_time_weight = b / max_beam_time
            
            # Calculate row intensity sum for this row
            row_intensity_sum = sum(intensity[i][j] for j in range(n))
            row_complexity = row_intensity_sum / (max_intensity * n) if max_intensity > 0 else 0.5
            
            # Complex constraints are weighted higher
            constraint_weight = 0.5 + 0.3 * beam_time_weight + 0.2 * row_complexity
            
            G.add_node(f'row_constraint_{i}_{b}', type=1, weight=min(constraint_weight, 1.0))
            
            # Connect to all cells in this row that could be affected by this beam time
            for j in range(n):
                if intensity[i][j] >= b:  # Cell needs at least this beam time
                    # Edge weight reflects how critical this beam time is for the cell
                    criticality = min(b / intensity[i][j], 1.0) if intensity[i][j] > 0 else 0.1
                    G.add_edge(f'cell_{i}_{j}', f'row_constraint_{i}_{b}', 
                              weight=criticality)
    
    # Add global beam time constraint (type 1)
    # This constraint relates total beam time to all shape matrices
    total_constraint_weight = 0.8  # High importance global constraint
    G.add_node('beamtime_constraint', type=1, weight=total_constraint_weight)
    
    # Connect all non-zero intensity cells to beam time constraint
    for i in range(m):
        for j in range(n):
            if intensity[i][j] > 0:
                # Weight by contribution to total beam time
                contribution = intensity[i][j] / total_intensity if total_intensity > 0 else 0.5
                G.add_edge(f'cell_{i}_{j}', 'beamtime_constraint', 
                          weight=min(contribution * 3.0, 1.0))
    
    # Add shape matrix count constraint (type 1)
    shape_constraint_weight = 0.7  # Important constraint for minimization
    G.add_node('shape_count_constraint', type=1, weight=shape_constraint_weight)
    
    # Connect all cells to shape count constraint (all cells participate)
    for i in range(m):
        for j in range(n):
            # Cells with higher intensity contribute more to shape matrix complexity
            if max_intensity > 0:
                intensity_contribution = intensity[i][j] / max_intensity
            else:
                intensity_contribution = 0.5
            
            # Non-linear weight for shape complexity
            shape_weight = 0.2 + 0.6 * math.sqrt(intensity_contribution)
            G.add_edge(f'cell_{i}_{j}', 'shape_count_constraint', 
                      weight=min(shape_weight, 1.0))
    
    # Add conflict edges between cells with very high intensity requirements
    # These compete for limited high-beam-time configurations
    high_intensity_threshold = max_intensity * 0.8 if max_intensity > 0 else 0
    high_intensity_cells = []
    
    for i in range(m):
        for j in range(n):
            if intensity[i][j] >= high_intensity_threshold:
                high_intensity_cells.append((i, j, intensity[i][j]))
    
    # Add conflict edges between high-intensity cells
    for idx1 in range(len(high_intensity_cells)):
        for idx2 in range(idx1 + 1, len(high_intensity_cells)):
            i1, j1, int1 = high_intensity_cells[idx1]
            i2, j2, int2 = high_intensity_cells[idx2]
            
            # Spatial distance affects conflict strength
            spatial_distance = abs(i1 - i2) + abs(j1 - j2)
            max_distance = m + n - 2
            
            # Nearby high-intensity cells create stronger conflicts
            if spatial_distance <= max_distance * 0.3:  # Only nearby cells
                distance_factor = 1.0 - (spatial_distance / max_distance)
                intensity_factor = (int1 + int2) / (2 * max_intensity) if max_intensity > 0 else 0.5
                
                # Non-linear conflict weight
                conflict_weight = distance_factor * intensity_factor * 0.8
                
                if conflict_weight > 0.2:  # Only add meaningful conflicts
                    G.add_edge(f'cell_{i1}_{j1}', f'cell_{i2}_{j2}', 
                              weight=min(conflict_weight, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()